;	$Id: dino-cnx.el,v 1.43 2004/03/03 12:48:37 vanicat Exp $	
;;; dino-msg-mode.el 

;;; Copyright (C) 1998  Vanicat Remi
;;;
;; Author: Remi Vanicat (vanicat@labri.u-bordeaux.fr)
;; Keywords: Dino 
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; A copy of the GNU General Public License can be obtained from 
;;; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 
;;; 02139, USA.
;;; http://www.gnu.org/copyleft/gpl.html

;;; Commentary:
;;;
;;; ce package est pour permetre la lecture de dino dans emacs
;;; ce fichier concerne la communication entre le serveur et le client
;;; 

(require 'dino-autoload)
(require 'cl)

(defcustom dino-login (let ((tmp (getenv "DINO_LOGIN")))
			(if tmp tmp
			  (getenv "OEUF_NAME")))
  "*le login"
  :type 'string
  :group 'dino)

(defcustom dino-nick (let ((tmp (getenv "DINO_NICK")))
		       (if tmp tmp
			 dino-login))
  "*le nick"
  :type 'string
  :group 'dino)

(defvar dino-cnx-load-hook ()
  "*Un hook lanc  la lecture de dino-cnx")

(defvar dino-tq ()
  "la transaction queue (tq2)")

(defvar dino-tq-lock ()
  "un lock pour la transaction queue")

(defvar dino-regexp-multireponse "^Err:[^\n]*\n\\|^Ok/n:[^\n]*\n\\(|[^\n]*\n\\)*#fin\n"
  "une regexp pour lire toutes les rponses possibles du serveur")
(defvar dino-regexp-unireponse "^Err:[^\n]*\n\\|^Ok/1:[^\n]*\n"
  "une regexp pour lire les reponses du serveur constitues d'une 
seule ligne")
(defvar dino-regexp-allreponse 
  "^Err:[^\n]*\n\\|^Ok/1:[^\n]*\n\\|^Ok/n:[^\n]*\n\\(|[^\n]*\n\\)*#fin\n"
  "une regexp pour lire les reponses du serveur constitues de 
plusieurs lignes")

(defvar dino-buffer ()
  "le buffer console (*dino*  priori)")

(defcustom dino-connect-anonymous (getenv "DINO_ANON")
  "dinel vas vous connecter de faons annonyme si non nil"
  :type 'boolean
  :group 'dino)

(defun dino-toggle-anonymous ()
  "change l'tat anonyme/non anonyme pour la prochaine reconnexion"
  (interactive)
  (setq dino-connect-anonymous (not dino-connect-anonymous)))

(defun dino-set-anonymous ()
  "met en tat anonyme (valable uniquement pour la prochaine reconnexion)"
  (interactive)
  (setq dino-connect-anonymous 't))

(defun dino-unset-anonymous ()
  "met en tat non anonyme (valable uniquement pour la prochaine reconnexion)"
  (interactive)
  (setq dino-connect-anonymous ()))

(defcustom dino-passwd (let ((tmp (getenv "DINO_PASS")))
			 (if tmp tmp
			   (getenv "OEUF_PASS")))
  "*le passwd"
  :type 'string
  :group 'dino)

(defvar dino-host (let ((tmp (getenv "DINO_SERV")))
		    (if tmp tmp
		      (setq tmp (getenv "OEUF_SERV"))
		      (if tmp tmp
			"server.dino.ens-lyon.org")))
  "*la machine pour le serveur")

(defcustom dino-hosts-list (list "server.dino.ens-lyon.org"
				 "gw1.dino.ens-lyon.org"
				 "gw2.dino.ens-lyon.org"
				 "gw3.dino.ens-lyon.org"
				 "gw4.dino.ens-lyon.org")
  "*la liste des serveurs (ou gateway)"
  :type '(repeat string)
  :group 'dino)

(defvar dino-hosts-list-maybe ()
  "liste des passerelles non testes")

(defvar dino-host-succes ()
  "non nil si une connexion a russi sur dino-host")

(defcustom dino-port 4030
  "le port du serveur"
  :type 'integer
  :group 'dino)

(defvar dino-connection ()
  "le process connection")

(defvar dino-check-message ()
  "si non nil, le client regarde rgulirement s'il y a un nouveau message
(non-implement pour l'instant)")

(defvar dino-waiting-server ""
  "if \"\" the server is not waiting, if \"w\" the server is waiting")


(defun dino-set-connect (login passwd host port)
  "tabli les diffrentes options de la connexion"
  (interactive (list (read-string "Login: " dino-login)
		     (dino-ask-passwd "Passwd: " dino-passwd)
		     (read-string "Host: " dino-host)
		     (read-minibuffer "Port: " (int-to-string dino-port))))
  (setq dino-login login)
  (setq dino-passwd passwd)
  (setq dino-host host)
  (setq dino-port port))

(defun dino-connect-direct ()
  (dino-connect-list))
;  "tabli la connexion avec les internal emacs"
;  (setq process-connection-type ())
;  (setq dino-connection 
;	(open-network-stream "dino" dino-buffer dino-host dino-port))
;  (setq process-connection-type t)
;  (setq dino-tq (tq2-create dino-connection 'dino-spurious 'dino-waiting-server))
;  (set-process-sentinel dino-connection 'dino-process-sentinel))

(defun dino-connect-host (dino-host)
  (setq dino-connection 
	(open-network-stream "dino" dino-buffer dino-host dino-port))
  (set-process-coding-system dino-connection 'iso-latin-9 'iso-latin-9))

(defun dino-connect-list ()
  (let ((connected ())
	(process-connection-type ())
	(host))
    (if (null dino-hosts-list-maybe)	; c'etait apres. Pourquoi ?
	(setq dino-hosts-list-maybe dino-hosts-list))
    (if (or dino-host-succes (null dino-hosts-list-maybe))
	(setq host dino-host)
      (setq host (nth (random (length dino-hosts-list-maybe))
		      dino-hosts-list-maybe))
      (setq dino-hosts-list-maybe (remove host dino-hosts-list-maybe)))
    (setq dino-host-succes ())
    (while (and host (not connected))
      (condition-case nil
	  (progn
	    (message (concat "Tentative de connexion  " host))
	    (dino-connect-host host)
	    (setq dino-host host)
	    (setq connected 't))
	((error quit)
	 (message (concat "La tentative de connexion  "
			  host " a echou"))))
      (setq host (nth (random (length dino-hosts-list-maybe))
		      dino-hosts-list-maybe))
      (setq dino-hosts-list-maybe (remove host dino-hosts-list-maybe)))
    (if (not connected)
	(error "Tous les serveurs (ou passerelles) ont echou"))
    (setq dino-tq
	  (tq2-create dino-connection 'dino-spurious 'dino-waiting-server))
    (set-process-sentinel dino-connection 'dino-process-sentinel)))

(defun dino-connect-next-serveur ()
  "reconnect avec un autre serveur (si possible)"
  (interactive)
  (dino-deconnect)
  (setq dino-host-succes ()))
	

;;;###autoload
(defun dino-lock (fn args)
  (if dino-tq-lock
      (setq dino-tq-lock 
	    (nconc dino-tq-lock '((dino-lock fn args))))
    (setq dino-tq-lock '(t))
    (apply fn args)))

;;;###autoload
(defun dino-unlock ()
  (let ((lock dino-tq-lock))
    (setq dino-tq-lock ())
    (mapcar (lambda (x) 
	      (if (consp x)
		  (apply (car x) (cdr x)))) (copy-list lock))))

;;;###autoload
(defun dino-send-command-unireponse (question closure fn)
  "ajoute une transaction a la transaction queue de dino"
;  (message question)
  (if dino-tq-lock
      (setq dino-tq-lock
	    (nconc dino-tq-lock 
		   (cons (list 'dino-send-command-unireponse
			       question closure fn) ())))
    (tq2-enqueue dino-tq question dino-regexp-unireponse closure fn)))

;;;###autoload
(defun dino-send-command-unireponse-in-lock (question closure fn)
  "ajoute une transaction a la transaction queue de dino"
;  (message question)
  (tq2-enqueue dino-tq question dino-regexp-unireponse closure fn))

;;;###autoload
(defun dino-send-command-multireponse (question closure fn)
  "ajoute une transaction a la transaction queue de dino"
;  (message question)
  (if dino-tq-lock
      (setq dino-tq-lock
	    (nconc dino-tq-lock 
		   (cons (list 'dino-send-command-multireponse
			       question closure fn) ())))
    (tq2-enqueue dino-tq question dino-regexp-multireponse closure fn)))

(defun dino-send-command-multireponse-in-lock (question closure fn)
  "ajoute une transaction a la transaction queue de dino"
;  (message question)
  (tq2-enqueue dino-tq question dino-regexp-multireponse closure fn))

(defun dino-send-command-allreponse (question closure fn)
  "ajoute une transaction a la transaction queue de dino"
;  (message question)
  (if dino-tq-lock
      (setq dino-tq-lock
	    (nconc dino-tq-lock 
		   (cons (list 'dino-send-command-allreponse
			       question closure fn) ())))
    (tq2-enqueue dino-tq question dino-regexp-allreponse closure fn)))

(defun dino-send-command-allreponse-in-lock (question closure fn)
  "ajoute une transaction a la transaction queue de dino"
;  (message question)
  (tq2-enqueue dino-tq question dino-regexp-allreponse closure fn))

(defun dino-connected (proc string) 
  (setq dino-host-succes 't)
  (dino-ecrit-recut proc string))

(defun dino-realy-connect () 
  "etabli la connexion"
  (message "dino connecting.....")
  (dino-connect-direct)
  (setq dino-tq-lock ())
  (tq2-enqueue dino-tq "" "[^\n]*\n\r?" dino-connection 
	       (function dino-ecrit-recut))
  (dino-erase-buffer)
  (set-marker (process-mark dino-connection) 0 
	      (process-buffer dino-connection))
  (dino-send-command-unireponse (concat (if dino-connect-anonymous 
					    "off_anonymous\n"
					  "off\n")
					dino-login "\n" 
					dino-passwd "\n")
				dino-connection 
				(function dino-connected))
  (dino-rc-serv-read-dinorc-maybe))

;;;###autoload
(defun dino-reconnect ()
  (if (or (null dino-connection)
	  (member (process-status dino-connection) 
		  (eval-when-compile (list 'exit 'signal 'closed nil))))
      (dino-realy-connect)))

(defun dino-connect () 
  (interactive) 
  (when (or (null dino-login) 
	    (null dino-port)
	    (null dino-passwd)
	    (and (null dino-hosts-list)
		 (null dino-host)))
    (setq dino-login (read-string "Login: " dino-login))
    (setq dino-passwd (dino-ask-passwd "Passwd: " dino-passwd))
    (setq dino-host (read-string "Host: " dino-host))
    (setq dino-port (read-minibuffer "Port: "
				     (int-to-string dino-port))))
  (when (null dino-host)
    (setq dino-host (car dino-hosts-list)))
  (setq  dino-buffer (dino-get-console-buffer))
  (when dino-tq (tq2-close dino-tq))
  (dino-connect-internall))

(defun dino-connect-internall ()
  "tabli la connexion"
  (dino-realy-connect)
  (dino-lit-nbmsg)
  (sleep-for 1)
  (accept-process-output)
  (message ""))


(defvar dino-proc-debug ())

;; sentinell
(defun dino-process-sentinel (process event)
  (tq2-close dino-tq)
  (dino-save-dinorc)
  (dino-ecrit-recut process event))

;; les fonction pour comuniquer avec le serveur
(defun dino-noop (proc string) (interactive) 
  "pour ne rien faire sur une communication" 
  ())

;;;###autoload
(defun dino-ecrit-recut (proc string) (interactive) 
  "pour crire le rsultat d'une commande dans le buffer *dino*"
  (let ((old-buffer (current-buffer)))
    (unwind-protect 
	(progn
	  (set-buffer (process-buffer dino-connection))
	  (let ((moving (= (point) (point-max)))
		(old-point (point)))
	    ;; Insert the text, moving the process-marker.
	    (goto-char (point-max))
	    (insert "msg :")
	    (message (substring string 0 -1))
	    (insert (substring string 0 -1) "\n")
	    (when moving (goto-char old-point))))
      (set-buffer old-buffer))))
      
(defun dino-send-command (str)
  "envoie commande STR au serveur,
traite la comande connect spcialement c'est  dire qu'il l'intercepte
et relance la connexion
Si la rponse est trop longue aller voir dans le buffer *dino*"
  (interactive (list (read-string "Commande  :")))
  (dino-reconnect)
  (if (not (string-equal str ""))
      (if (and (>=  (length str) 7) 
	       (string-equal (substring str 0 7) "connect"))
	  (dino-connect)
	(dino-send-command-allreponse (concat str "\n")
				      dino-connection
				      (function dino-ecrit-recut)))))


(defun dino-coordonne-apply (funct &rest arg)
  (dino-send-command-unireponse "nbmsg\n"
				(cons funct arg)
				(function 
				 (lambda (x y) (apply (car x) (cdr x))))))

(defun dino-spurious (str tq2)
  (dino-ecrit-recut (tq2-process tq2) str)
  (if (string-equal str "Err: temps depasse\n")
      (tq2-close tq2)))


(defun dino-deconnect () 
  (interactive)
  "pour se dconnecter du serveur"
  (if (not (member (process-status dino-connection)
		   '(exit signal closed ())))
      (dino-send-command-unireponse "quit\n" 
				    dino-connection
				    (function dino-ecrit-recut))))

(run-hooks 'dino-cnx-load-hook)
(provide 'dino-cnx)

