;;; tq2.el --- utility to maintain a transaction queue
;;; it's only tq.el whith some modification

;; Copyright (C) 1985, 1986, 1987, 1992 Free Software Foundation, Inc.

;; Author: Scott Draves <spot@cs.cmu.edu>
;; Adapted-By: ESR
;; Keywords: extensions
;; modified by: Remi Vanicat <vanicat@labri.u-bordeaux.fr>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; A copy of the GNU General Public License can be obtained from 
;;; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 
;;; 02139, USA.
;;; http://www.gnu.org/copyleft/gpl.html

;;; Commentary:
 
;;; manages receiving a stream asynchronously, 
;;; parsing it into transactions, and then calling
;;; handler functions

;;; Our basic structure is the queue/process/buffer triple.  Each entry
;;; of the queue is a regexp/closure/function triple.  We buffer
;;; bytes from the process until we see the regexp at the head of the
;;; queue.  Then we call the function with the closure and the
;;; collected bytes.

;;; modification by Remi Vanicat on Aug 12 1998
;;; modification : add of a unwind-protection to tq2-filter
;;;                add of a possible configuration of traitment 
;;;                    of spurious communication

(require 'cl)

;;; Code:

;;;###autoload
(defun tq2-create (process &optional func watcher)
  "Create and return a transaction queue communicating with PROCESS.
PROCESS should be a subprocess capable of sending and receiving
streams of bytes.  It may be a local process, or it may be connected
to a tcp server on another machine.
the optional func is for the traitment of spirituous"
  (let ((tq2 (list () 
		   process
		   (generate-new-buffer
		    (concat " tq2-temp-" (process-name process)))
		   (if func func 'tq2-spurious-default)
		   watcher)))
    (set-process-filter process
			(`(lambda (proc string)
			   (tq2-filter  '(, tq2) string))))
    tq2))

;;; accessors
(defun tq2-queue    (tq2) (car tq2))
;;;###autoload
(defun tq2-process  (tq2) (cadr tq2))
(defun tq2-buffer   (tq2) (caddr tq2))
(defun tq2-spurious (tq2) (cadddr tq2))
(defun tq2-watcher  (tq2) (cadddr (cdr tq2)))

(defun tq2-set-waiting (tq2)
  (let ((watcher (tq2-watcher tq2)))
    (if watcher
	(set-variable watcher 't))))

(defun tq2-unset-waiting (tq2)
  (let ((watcher (tq2-watcher tq2)))
    (if watcher
	(set-variable watcher ()))))

(defun tq2-queue-add (tq2 re closure fn)
  (setcar tq2 (nconc (tq2-queue tq2)
		    (cons (cons re (cons closure fn)) nil)))
  'ok)

(defun tq2-queue-head-regexp  (tq2) (car (car (tq2-queue tq2))))
(defun tq2-queue-head-fn      (tq2) (cdr (cdr (car (tq2-queue tq2)))))
(defun tq2-queue-head-closure (tq2) (car (cdr (car (tq2-queue tq2)))))
(defun tq2-queue-empty        (tq2) (not (tq2-queue tq2)))
(defun tq2-queue-pop          (tq2) (setcar tq2 (cdr (car tq2))) (null (car tq2)))
 

;;; must add to queue before sending!
;;;###autoload
(defun tq2-enqueue (tq2 question regexp closure fn)
  "Add a transaction to transaction queue TQ2.
This sends the string QUESTION to the process that TQ2 communicates with.
When the corresponding answer comes back, we call FN
with two arguments: CLOSURE, and the answer to the question.
REGEXP is a regular expression to match the entire answer;
that's how we tell where the answer ends."
  (tq2-queue-add tq2 regexp closure fn)
  (tq2-set-waiting tq2)
  (process-send-string (tq2-process tq2) question))

;;;###autoload
(defun tq2-close (tq2)
  "Shut down transaction queue TQ2, terminating the process."
  (tq2-unset-waiting tq2)
  (delete-process (tq2-process tq2))
  (kill-buffer (tq2-buffer tq2)))

(defun tq2-filter (tq string)
  "Append STRING to the TQ's buffer; then process the new data."
  (let ((old-buffer (current-buffer)))
    (unwind-protect
	(progn
	  (set-buffer (tq2-buffer tq))
	  (goto-char (point-max))
	  (insert string)
	  (tq2-process-buffer tq))
      (set-buffer old-buffer)
      (if (null (tq2-queue tq)) (tq2-unset-waiting tq)))))

(defun tq2-process-buffer (tq2)
  "Check TQ2's buffer for the regexp at the head of the queue."
  (let ((fini . ()))
    (while (not fini)
      (setq fini 't)
      (set-buffer (tq2-buffer tq2))
      (if (= 0 (buffer-size)) ()
	(if (tq2-queue-empty tq2)
	    (let ((str (buffer-string)))
	      (delete-region (point-min) (point))
	      (apply (tq2-spurious tq2) (list str tq2)))
	  (goto-char (point-min))
	  (if (re-search-forward (tq2-queue-head-regexp tq2) nil t)
	      (let ((answer (buffer-substring (point-min) (point)))
		    (fn (tq2-queue-head-fn tq2))
		    (closure (tq2-queue-head-closure tq2)))
		(delete-region (point-min) (point))
		(tq2-queue-pop tq2)
		(funcall fn closure answer)
		(setq fini ()))))))))

(defun tq2-spurious-default (str tq2)
  "default spurious traitment"
  (let ((buf (generate-new-buffer "*spurious*")))
    (set-buffer buf)
    (insert str)
    (pop-to-buffer buf nil)
    (error (concat "Spurious communication from process "
		   (process-name (tq2-process tq2))
		   ", see buffer *spurious*."))))
(provide 'tq2)

;;; tq2.el ends here
