/* conv.c

   Ce programme peut être librement distribué suivant les termes de la
   « General Public License », version 2 ou ultérieure. Voir le fichier
   COPYING pour les détails. Si vous n'avez pas reçu de copie de cette
   licence avec le programme, allez voir le site www.gnu.org pour l'obtenir.

   This program can be freely distributed following the terms of the
   General Public Licence, version 2 or later. See file COPYING for
   details. If you haven't received a copy of this license with the
   program, please visit www.gnu.org to obtain it.

*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <libintl.h>
#include <wchar.h>
#include <errno.h>
#include "defs.h"
#ifdef HAVE_ICONV
#include <langinfo.h>
#include <iconv.h>
#include <locale.h>
#endif
#include "conv.h"
#include "init.h"
#include "utf8.h"

static char *suffixes[]={"com","org","net","fr","de","uk","tw","asso.fr","co.tw","co.uk"};


/* conversion

   Convertit un message d'un type vers un autre, selon ce que vaut type.
   Renvoie un truc malloué.
   Si type est à 0, on ne convertit rien.
   Si type vaut 1, on fait du HTML brut -> HTML simplifié.
   Si type vaut 2, on fait du HTML brut -> texte simple.
   Si type vaut -1, on fait du HTML simplifié -> HTML.
   S'il vaut -2, on fait du texte simple -> HTML.
   Si rep est non nul, on rajoute les "> " au début de chaque ligne, en coupant de façon sûre à 80 colonnes.
*/

char *conversion(const char *message,int type,int rep)
{
  char *tampon;
  int i=0,j=0,k,c,nbsuff;
  int zob,zub=0,ncarac,flag_tag=0,flag_prout=0;
  int flag_ahref=0;

  tampon=malloc(2*BUFSIZE);

  while(message[i] && j<BUFSIZE)
    {
      int debut_ligne=j;
      c=0;
      if(rep)
	{
	  tampon[j++]='>';
	  tampon[j++]=' ';
	  c=2;
	}
      while(message[i]!='\n' && message[i])
	{
          if (rep!=0 && c>=80)
	    { /* On fait une césure */
	      int prev_space=0;
	      int nb_char_rep=0;
	      int j2=debut_ligne;
	      /* Compter la longueur de "(> ?)*> " existant au début de la ligne en
	       * cours */
	      for(;;)
		{
		  int j3=j2;
		  int nb_chevrons=0;
		  /* On passe les chevrons */
		  while (j3 < j-1 && tampon[j3]=='>')
		    {
		      nb_chevrons ++;
		      j3++;
		    }
                  /* On vérifie qu'on a bien une espace après au moins un chevron... */
		  if (nb_chevrons==0 || j3==j || tampon[j3]!=' ')
		    {
		      /* ...sinon, on part */
		      break;
		    }
		  nb_char_rep += nb_chevrons + 1;
		  j2 = j3 + 1;
		}
	      /* Trouver la dernière espace de la ligne */
	      while (j2<j)
		{
	          if (tampon[j2]==' ')
		    {
	              prev_space=j2;
	            }
	          j2++;
	        }
              if (prev_space != 0)
		{
	          /* On a trouvé où couper */
                  tampon[prev_space]='\n';
                  /* On fait de la place pour insérer les signes '> ' de la ligne précédente */
	          memmove(&tampon[prev_space+1+nb_char_rep], &tampon[prev_space+1], j-(prev_space+1));
	          /* On recopie les '> ' */
	          memmove(&tampon[prev_space+1], &tampon[debut_ligne], nb_char_rep);
	          /* On positionne les différents compteurs correctement */
	          j += nb_char_rep;
	          c = nb_char_rep + j-(prev_space+1);
	          debut_ligne=prev_space+1;
	        }
	      /* Et sinon tant pis, on ne coupe pas */
	    }
	  switch(type)
	    {
	    case 0:
	      switch(message[i])
		{
		case '&':
		  flag_prout=1;
		  break;
		case ';':
		  if(flag_prout)
		    flag_prout=0;
		  if(!flag_tag)
		    c++;
		  break;
		case '<':
		  flag_tag=1;
		  break;
		case '>':
		  if(flag_tag)
		    {
		      flag_tag=0;
		      break;
		    } // Pas de break ici, c'est un symbole courant.
		default:
		  if(flag_tag==0 && flag_prout==0)
		    c++;
		}
	      tampon[j++]=message[i++];
	      break;
	    case 1: // HTML => HTML simplifié
	      if(message[i]=='<')
		{
		  zob=0; // Est-ce que c'est une balise reconnue ?
		  switch(message[i+1])
		    {
		    case '/':
		      switch(message[i+2])
			{
			case 'b':
			case 'B':
			case 'i':
			case 'I':
			case 'u':
			case 'U':
			case 's':
			case 'S':
			case 'a':
			case 'A':
			  if(message[i+3]=='>')
			    zob=1;
			  break;
			}
		      break;
		    case 'b':
		    case 'B':
		    case 'i':
		    case 'I':
		    case 'u':
		    case 'U':
		    case 's':
		    case 'S':
		      if(message[i+2]=='>')
			zob=1;
		      break;
		    case 'a':
		    case 'A':
		      if(strncasecmp(message+i,"<a href=\"",9)==0)
			zob=1;
		      break;
		    }
		  if(zob)
		    {
		    tampon[j++]=message[i++];
		    flag_tag=1;
		    }
		  else
		    while(message[i] && message[i++]!='>');
		}
	      else if(flag_tag==0 && message[i]=='&')
		{
		  if(strncasecmp(message+i,"&nbsp;",6)==0)
		  {
		    /* NBSP en UTF-8 */
		    tampon[j++]=0xc2;
		    tampon[j++]=0xa0;
		  }
		  else if(strncasecmp(message+i,"&lt;",4)==0)
		    {
		      if((strchr("aAbBiIsSuU",message[i+4])!=NULL && message[i+5]=='>') || (message[i+4]=='/' && strchr("aAbBiIsSuU",message[i+5])))
			j+=sprintf(tampon+j,"&lt;");
		      else
			tampon[j++]='<';
		    }
		  else if(strncasecmp(message+i,"&gt;",4)==0)
		    tampon[j++]='>';
		  else if(strncasecmp(message+i,"&amp;",5)==0)
		    {
		      if(strncasecmp(message+i+5,"nbsp;",5)==0 || strncasecmp(message+i+5,"lt;",3)==0 || strncasecmp(message+i+5,"gt;",3)==0 || strncasecmp(message+i+5,"amp;",4)==0)
			j+=sprintf(tampon+j,"&amp;");
		      else
			tampon[j++]='&';
		    }
		  while(message[i] && message[i++]!=';');
		  c++;
		}
	      else
		{
		  if(flag_tag==0)
		    c++;
		  if(message[i]=='>')
		    flag_tag=0;
		  tampon[j++]=message[i++];
		}
	      break;
	    case 2: // HTML-> texte simple
	      switch(message[i])
		{
		case '<':
		  while(message[i] && message[i++]!='>');
		  break;
		case '&':
		  if(strncasecmp(message+i,"&amp;",5)==0)
		    tampon[j++]='&';
		  else if(strncasecmp(message+i,"&lt;",4)==0)
		    tampon[j++]='<';
		  else if(strncasecmp(message+i,"&gt;",4)==0)
		    tampon[j++]='>';
		  else if(strncasecmp(message+i,"&nbsp;",6)==0)
		  {
		    /* NBSP en UTF-8 */
		    tampon[j++]=0xc2;
		    tampon[j++]=0xa0;
		  }
		  while(message[i] && message[i++]!=';');
		  c++;
		  break;
		default:
		  tampon[j++]=message[i++];
		  c++;
		  break;
		}
	      break;
	    case -1: //HTML simplifié -> HTML
	      switch(message[i])
		{
		case '&':
		  if(flag_tag || strncasecmp(message+i,"&amp;",5)==0 || strncasecmp(message+i,"&lt;",4)==0 || strncasecmp(message+i,"&gt;",4)==0 || strncasecmp(message+i,"&nbsp;",6)==0)
		    {
		      flag_prout=1;
		      tampon[j++]=message[i++];
		    }
		  else
		    {
		      j+=sprintf(tampon+j,"&amp;");
		      i++;
		      c++;
		    }
		  break;
		case '<':
		  if((strchr("bBiIuUsS",message[i+1]) && message[i+2]=='>') || (message[i+1]=='/' && strchr("bBiIuUsS",message[i+2]) && message[i+3]=='>'))
		    tampon[j++]=message[i++];
		  else if(strncasecmp(message+i,"<a href=\"",9)==0)
		    {
		      flag_ahref=1;
		      flag_tag=1;
		      tampon[j++]=message[i++];
		    }
		  else if(strncasecmp(message+i,"</a>",4)==0)
		    {
		      flag_ahref=0;
		      tampon[j++]=message[i++];
		    }
		  else
		    {
		      j+=sprintf(tampon+j,"&lt;");
		      i++;
		      c++;
		    }
		  break;
		case ';':
		  flag_prout=0;
		default:
		  // Parsing des URLs.
		  zub=i;
		  if(flag_ahref==0 && (i==0 || message[i-1]!='\"') &&
		      (  (strncmp(message+i,"http://",7)==0 && (zub=i+7))
		     || (strncmp(message+i,"www.",4)==0 && message[i+4]>='a' && message[i+4]<='z')
		     || (strncmp(message+i,"ftp",3)==0 && ((message[i+3]=='.' && message[i+4]>='a' && message[i+4]<='z') || (message[i+3]==':' && message[i+4]=='/' && message[i+5]=='/' && (zub=i+6))))
			 ))
		    {
		      zob=i;
		      while(1)
			{
			  if(message[i]==' ' || message[i]=='\n' || message[i]==')' || message[i]=='>' || ((message[i]==',' || message[i]=='.') && (message[i+1]==' ' || message[i+1]=='\n' || message[i+1]==0) ))
			    break;
			  else
			    i++;
			}
		      j+=sprintf(tampon+j,"<a href=\"");
		      strncpy(tampon+j,message+zob,i-zob);
		      j=j+i-zob;
		      
		      // On extrait le nom du site.
		      while(strncmp(message+zub,"www.",4)==0 || strncmp(message+zub,"ftp.",4)==0)
			zub+=4;
		      nbsuff=sizeof(suffixes)/sizeof(char *);
		      ncarac=0;
		      j+=sprintf(tampon+j,"\">&lt;");
		      while(zub<i && message[zub]!='/')
			{
			  if(message[zub]=='.')
			    {
			      for(k=0;k<nbsuff;k++)
				if(strncmp(message+zub+1,suffixes[k],strlen(suffixes[k]))==0)
				  break;
			      if(k<nbsuff)
				break;
			    }
			  tampon[j++]=message[zub++];
			  ncarac++;
			}
		      if(ncarac<3) // Le nom trouvé est trop court.
			{
			  j-=ncarac;
			  ncarac=sprintf(tampon+j,_("link"));
			  j+=ncarac;
			}
		      if(ncarac>20) // Là, il est trop long.
			{
			  j=j-ncarac+16;
			  ncarac=16+sprintf(tampon+j,"...");
			  j=j+ncarac-16;
			}
		      j+=sprintf(tampon+j,"></a>");
		      c+=ncarac+2;
		    }
		  else
		    {
		      if(flag_tag==0 && flag_prout==0)
			c++;
		      if(flag_tag && message[i]=='>')
			flag_tag=0;
		      tampon[j++]=message[i++];
		    }
		}
	      break;
	    case -2: // texte -> HTML
	      switch(message[i])
		{
		case '<':
		  j+=sprintf(tampon+j,"&lt;");
		  break;
		case '&':
		  j+=sprintf(tampon+j,"&amp;");
		  break;
		default:
		  tampon[j++]=message[i];
		}
	      c++;
	      i++;
	      break;
	    }
	}
      if(message[i]=='\n')
	{ /* le message avait un retour à la ligne */
	  tampon[j++]='\n';
	  i++;
	}
    }
  tampon[j++]=0;
  tampon=realloc(tampon,j);
  return tampon;
}

#ifndef HAVE_ICONV
#  warning iconv support not found by configure.
#  warning Expect problems if the terminal is not in latin9
#else
/* charset du serveur (UTF-8) vers locale courante et vice-versa, renvoient le nombre de
 * chars écris */
iconv_t scribus=(iconv_t)(-1);
iconv_t subircs=(iconv_t)(-1);
iconv_t cd_iso_8859_15=(iconv_t)(-1);
iconv_t cd_utf8=(iconv_t)(-1);
static size_t grr(const char *ga, size_t bu, char *zo, size_t meu) {
  if (meu<bu)
    bu = meu;
  strncpy(zo,ga,bu);
  return bu;
}
size_t scribuser(const char *ga, size_t bu, char *zo, size_t meu) {
  char *sga = (char *)ga, *szo = zo;
  wchar_t zorglub;
  mbstate_t bulgroz;
  size_t touca;
  while(1) {
    if (iconv(scribus,&sga,&bu,&szo,&meu)!=(size_t)(-1) || errno == E2BIG)
      goto out;
    memset(&bulgroz, 0, sizeof(bulgroz));
    if ((touca = mbrtowc(&zorglub, sga, bu, &bulgroz)) != (size_t)(-1)) {
      switch (zorglub) {
        case 0x02BC:
        case 0x2019:
          *szo++ = '\'';
	  break;
	default:
          *szo++ = '?';
	  break;
      }
      sga+=touca;
      if (!(bu-=touca) || !--meu)
	goto out;
      continue;
    }
    do {
      *szo++ = '?';
      sga++;
      if (!(bu--) || !--meu)
	goto out;
      memset(&bulgroz, 0, sizeof(bulgroz));
    } while (mbrtowc(&zorglub, sga, bu, &bulgroz) == (size_t)(-1));
  }
out:
  return szo-zo;
}
size_t subircser(const char *ga, size_t bu, char *zo, size_t meu) {
  char *sga = (char *) ga, *szo = zo;
  /* dans ce sens, il ne devrait jamais y avoir de problème */
  if (iconv(subircs,&sga,&bu,&szo,&meu)==(size_t)(-1))
    return grr(ga,bu,zo,meu);
  return szo-zo;
}

/* essayer de toujours arriver à traduire */
#define CREDO "//TRANSLIT"
#define CHARSET "UTF-8"
void amen(void)
{
  char *maria = setlocale(LC_CTYPE,NULL);
  char *pater = nl_langinfo(CODESET);
  /* conversion des messages avec l'encodage par défaut du serveur en UTF-8 */
  cd_iso_8859_15 = iconv_open(CHARSET CREDO, "ISO-8859-15");
  if (cd_iso_8859_15 == (iconv_t)(-1))
    cd_iso_8859_15 = iconv_open(CHARSET, "ISO-8859-15");
  /* vérification des messages UTF-8 du serveur */
  cd_utf8 = iconv_open(CHARSET, "UTF-8");
  /* si aucune locale n'est indiquée, on suppose qu'on peut ne pas traduire et
   * que ça ira... */
  if (maria && strcmp(maria, "C") && strcmp(pater,CHARSET)) {
    if ((scribus = iconv_open(CHARSET CREDO, "")) == (iconv_t)(-1)) {
      if ((scribus = iconv_open(CHARSET, "")) == (iconv_t)(-1))
	return;
    }
    if ((subircs = iconv_open(CREDO,CHARSET)) != (iconv_t)(-1)
      ||(subircs = iconv_open("",CHARSET)) != (iconv_t)(-1))
      return;
    iconv_close(scribus);
    scribus = (iconv_t)(-1);
  }
}

static
size_t strcpy_iconv_realloc_grow(iconv_t cd, char**from, size_t inlength,
                            char **init_to, size_t skip, size_t outlength,
                            size_t offset, size_t *grow) {
  size_t ret;
  size_t init_outlength=outlength;
  char *to = *init_to;
  if (cd == (iconv_t)(-1)) {
    return strcpy_noiconv_realloc(cd, from, inlength, init_to, skip, outlength, offset);
  }
  to += skip;
  do {
    ret=iconv(cd, from, &inlength, &to, &outlength);
    if (ret != (size_t)(-1))
      return init_outlength-outlength;
    switch(errno) {
    case EILSEQ:
      /* La conversion a échouée, on va tenter d'interpréter le caractère
       * comme du latin-15 si on n'est pas dans ce mode là */
      if (cd == cd_iso_8859_15) {
        /* On ne sait plus quoi faire... */
        (*from) ++; inlength--;
        if (outlength>0) {
          *to='?';
          to++; outlength--;
        }
      } else {
        if ((debug & DEBUG_ENCODING) && outlength>0) {
          *to='{';
          to++; outlength--;
        }
        int newskip= to - *init_to;
	size_t grow=0;
        int outwrote=strcpy_iconv_realloc_grow(cd_iso_8859_15, from, 1,
		init_to, newskip, outlength, offset, &grow);
        inlength--;
	outlength += (grow - outwrote);
	init_outlength += grow;
        to=*init_to + skip + (init_outlength - outlength);
        if ((debug & DEBUG_ENCODING) && outlength>0) {
          *to='}';
          to++; outlength--;
        }
      }
      break;
    case EINVAL:
      if (outlength>0) {
        *to='*';
        to++; outlength--;
      }
      return init_outlength-outlength;
      break;
    case E2BIG:
      *init_to=realloc(*init_to, skip+init_outlength+offset+10);
      if (*init_to == NULL) {
        return init_outlength-outlength;
      }
      init_outlength += 10;
      outlength += 10;
      (*grow) += 10;
      to=*init_to + skip + (init_outlength - outlength);
      break;
    default:
      perror("iconv");
      return init_outlength-outlength;
    }
  } while(1);
}

size_t strcpy_iconv_realloc(iconv_t cd, char**from, size_t inlength,
                            char **init_to, size_t skip, size_t outlength,
                            size_t offset) {
  size_t grow=0;
  return strcpy_iconv_realloc_grow(cd, from, inlength,
	   init_to, skip, outlength, offset, &grow);
}
#endif
