(* $Id: connect.ml,v 1.16 2001/06/16 17:07:16 mjambon Exp $ *)
open Unix
open ThreadUnix (* overrides some definitions of module Unix *)
open Printf

open Dodo
open Dodo_settings
open Com
open Login

exception Slow_connection
let timeout = float Default.timeout
let sec = if Default.timeout > 1 then "secondes" else "seconde"


let prn s = printf "%s\n" s; flush Pervasives.stdout

let success = ref false
let succeeded () = !success

exception Shorter_string of string * int
let rec read_string in_chan s =
  let len = String.length s in
  let rec loop i =
    if i < len then
      let c = input_char in_chan in
      if c = s.[i] then
	loop (i+1)
      else
	(*raise (Shorter_string (s,i))*) 
	(eprintf "`%c'" c; flush Pervasives.stderr) in
  loop 0

let skip_char in_chan n =
  for i = 1 to n do ignore (input_char in_chan) done

let echo_line in_chan =
  print_string (input_line in_chan);
  print_newline ()

let eight_chars s =
  let len = String.length s in
  let s' = String.make 8 '\000' in
  if len < 8 then
    String.blit ~src:s ~src_pos:0 ~dst:s' ~dst_pos:0 ~len
  else
    String.blit ~src:s ~src_pos:0 ~dst:s' ~dst_pos:0 ~len:8;
  s'


(* dummy initialization *)
let dino_outchan = ref Pervasives.stdout

let dino_connection = ref false
let on () = !dino_connection
let off () = not !dino_connection

(* Queue of server responses, built by one listening thread
   and destroyed by other threads: *)
let response_queue = (Queue.create () : response Queue.t)
(*let queue_mutex = Mutex.create ()*)

let dqueue_mutex = Mutex.create ()
let some_query_mutex = Mutex.create ()
let some_response_mutex = Mutex.create ()

let some_query_to_send = Condition.create ()
let some_response = Condition.create ()


let time_var () =
  let time_ref = ref (Unix.time ()) in
  let m = Mutex.create () in
  let set () =
    Mutex.lock m;
    time_ref := Unix.time ();
    Mutex.unlock m
  and get () =
    Mutex.lock m;
    let t = !time_ref in
    Mutex.unlock m;
    t in
  (set,get)
    

(*let (set_last_query, get_last_query) = time_var ()*)
let (set_last_response, get_last_response) = time_var ()

let add_response response =
  set_last_response ();
  Mutex.lock some_response_mutex;
  Queue.add response response_queue;
  Condition.signal some_response;
  Mutex.unlock some_response_mutex

let multi_add_response response n =
  set_last_response ();
  Mutex.lock some_response_mutex;
  for i = 1 to n do
    Queue.add response response_queue
  done;
  Condition.signal some_response;
  Mutex.unlock some_response_mutex




let dino_listen inchan = (* high-level listener *)
  try
    while !dino_connection do
      (* All this stuff is defined in Com *)
      let response = read_response inchan in
      match response with
	  Must_reconnect ->
	    Tk_info.display "Le serveur nous a dconnects";
	    dino_connection := false
	      (* not added to the queue, since unexpected *)
(*	| Any_response s -> 
	    Tk_info.display s
	      (* not added to the queue, since unexpected *)*)
	| _ -> add_response response
    done
  with
      End_of_file -> dino_connection := false (* R.I.P. *)


let dino_start_listen inchan =
  ignore (Dodo_thread.create dino_listen inchan)


let at_first_connection = ref (None : (unit->unit) option)

let rec dino_connect () =
  try
    begin
      let (hostname,port) = Dodo_settings.get_server ()
      and (login,passwd) = Login.get () in
      
      let addr_array = (gethostbyname hostname).h_addr_list
			 (* may raise Not_found *) in
      if Array.length addr_array = 0 then
	failwith (Printf.sprintf "host %s unreachable" hostname)
      else
	let addr = addr_array.(0) in
	let sockaddr = ADDR_INET (addr, port) in
	let (inchan, outchan) = 
	  try open_connection sockaddr
	  with _ -> raise (No_response (hostname, port)) in

	let infd = descr_of_in_channel inchan in
	let welcome = input_line inchan in (* Welcome-message *)

	let send_line s = 
	  Printf.fprintf outchan "%s\n" s; flush outchan in
	
	let passwd8 = eight_chars passwd in
	
	if !Dodo_settings.anonymous then send_line "off_anonymous"
	else send_line "off";
	send_line login;
	send_line passwd8;
	
	let first_response = input_line inchan in
	if String.sub first_response ~pos:1 ~len:3 = "Err" then
	  (Login.login_window (); raise Reconnect_later);
	
	(*    print first_response;*)
	Tk_info.display (sprintf "Vous tes connect  %s sur le port %i" 
			   hostname port);
	dino_connection := true;
	success := true;
	clear_failures ();
	dino_outchan := outchan;
	dino_start_listen inchan; (* high-level listening thread *)
	outchan
    end
  with
      Not_found ->
	let ((hostname,port) as serv_port) = get_server () in
	let err = sprintf "Hte %s inconnu" hostname in
	Tk_info.display err;
	dodo_error err;
	add_failure serv_port;
	let next = try_another_server () in
	if connection_failures next = 0 then
	  dino_connect ()
	else
	  (Login.server_window (); raise Reconnect_later)

    | No_response (hostname,port) -> 	
	let err = sprintf "Le serveur %s ne rpond pas (port %i)"
		    hostname port in
	Tk_info.display err;
	dodo_error err;
	add_failure (hostname,port);
	let next = try_another_server () in
	if connection_failures next = 0 then
	  dino_connect ()
	else
	  (Login.server_window (); raise Reconnect_later)

    | End_of_file ->
	let ((hostname,port) as serv_port) = get_server () in
	let err = sprintf "Le serveur %s a coup la connexion (port %i)"
		    hostname port in
	Tk_info.display err;
	dodo_error err;
	add_failure (hostname,port);
	let next = try_another_server () in
	if connection_failures next = 0 then
	  dino_connect ()
	else
	  raise Reconnect_later
  

let dino_send_list l =
  let valid_outchan = 
    if !dino_connection then
      !dino_outchan
    else
      dino_connect () in
  List.iter ~f:(output_string valid_outchan) l;
  flush valid_outchan


(* Implementation of high-level commands *)

(* Queue of commands that are waiting to be sent and treated by the server *)
type dino_query = { qry_lines : string list; (* list of strings to send *)
		    qry_error_handler : exn -> unit;
		    mutable qry_state : [`Not_sent|`Sent];
		    qry_actions : (response -> unit) list }
		    
let send_request query = 
  Mutex.unlock dqueue_mutex;
  (* the queue is unlocked during the sending period *)
  try
    (dino_send_list query.qry_lines; (* may raise Reconnect_later *)
     query.qry_state <- `Sent;
     Mutex.lock dqueue_mutex; (* the queue is relocked *)
     query.qry_actions)
  with exn -> 
    query.qry_state <- `Sent;
    Mutex.lock dqueue_mutex; (* the queue is relocked *)
    query.qry_error_handler exn;
    []


let query_queue = Double_queue.create ~transition:send_request


let response_thread_fun () =
  let rec take_response () =
    let rec get_response () = 
      try (let response = Queue.take response_queue in
	   Mutex.unlock dqueue_mutex;
	   response)
      with Queue.Empty -> 
	Mutex.unlock dqueue_mutex;
	Condition.wait some_response ~locking:some_response_mutex;
	get_response () in
    Mutex.lock some_response_mutex;
    let response = get_response () in
    Mutex.unlock some_response_mutex;

    Mutex.lock dqueue_mutex;
    (try
       let pending_action = 
	 Double_queue.take query_queue in
       Mutex.unlock dqueue_mutex;
       pending_action response
     with Queue.Empty ->
       Mutex.unlock dqueue_mutex);
    
    take_response () (* loop *) in

  try take_response () with 
      exn -> 
	Mutex.unlock some_response_mutex; 
	Mutex.unlock dqueue_mutex;
	raise exn


let sending_thread_fun () =
  let rec loop () =
    let rec send_query () =
      Mutex.lock dqueue_mutex;
      try
	(Double_queue.transfer query_queue;
	 Mutex.unlock dqueue_mutex;
	 (*set_last_query ()*))

      with Queue.Empty -> 
	Mutex.unlock dqueue_mutex;
	Condition.wait some_query_to_send ~locking:some_query_mutex;
	send_query () in
    Mutex.lock some_query_mutex;
    send_query ();
    Mutex.unlock some_query_mutex;
    loop () in

  try loop () 
  with exn -> 
    Mutex.unlock some_query_mutex;
    Mutex.unlock dqueue_mutex;
    raise exn


(* Start of both threads *)
let response_thread = Dodo_thread.create response_thread_fun ()
let sending_thread = Dodo_thread.create sending_thread_fun ()

let default_error_handler = fun (exn : exn) -> ()

let dino_send_command 
  ?(error_handler = default_error_handler)
  command actions =
  (* lines : list of arbitrary strings to send;
     actions : list of actions (the first one is applied to the 
     first response, the second one is applied to the second response,
     and so on. *)

(*  print_string "# ";
  List.iter ~f:print_string command;
  flush Pervasives.stdout;*)

  if !dino_connection || not !eco then
    let query = { qry_lines = command;
		  qry_error_handler = error_handler;
		  qry_state = `Not_sent;
		  qry_actions = actions } in
    
    Mutex.lock some_query_mutex;
    Mutex.lock dqueue_mutex;
    Double_queue.add query query_queue;
    Mutex.unlock dqueue_mutex;
    
    Condition.signal some_query_to_send;
    Mutex.unlock some_query_mutex

  else
    let response = 
      Error "Connectez-vous explicitement (vous tes en mode conomique)" in
    (* Hum hum, not really efficient... *)
    ignore (Dodo_thread.create (fun () ->
			     List.iter ~f:(fun f -> f response) actions) ())



(* Text mode *)
exception Not_empty
let not_empty s =
  try
    (for i = 0 to String.length s - 1 do
       if int_of_char s.[i] > 32 then raise Not_empty
     done;
     false)
  with Not_empty -> true

let timeout_exit () =
  eprintf "Impossible de se connecter dans les dlais (%i %s).
Veuillez utiliser l'option -eco si vous voulez utiliser Dodo hors connexion.
" Default.timeout sec;
  flush Pervasives.stderr;
  exit 1


let timed_dino_connect ?(quit=true) () = 
  (* It's use once produced segmentation faults *)
  let exn = ref None in
  let t1 = Unix.time () in
  ignore (Dodo_thread.create (fun () -> try ignore (dino_connect ())
			 with e -> exn := Some e) ());
  while off () &&  Unix.time () -. t1 < timeout do
    Thread.delay 0.1
  done;
  if off () then 
    if quit then
      timeout_exit ()
    else raise Slow_connection;

  match !exn with None -> () | Some e -> raise e


  
let force_disconnect () =
  Tk_info.display "Dconnexion en cours...";
  dino_send_command 
    ["quit\n"] 
    [(fun r -> 
	Tk_info.display "Vous tes  prsent dconnect")];
  dino_connection := false

let disconnect () = 
  if !dino_connection then
    force_disconnect ()
  else
    Tk_info.display "Vous tiez dj dconnect"


let connect () = 
  if !dino_connection then
    Tk_info.display "Vous tes dj connect"
  else
    try
      (Tk_info.display "Connexion en cours...";
       ignore (dino_connect ());
       (match !at_first_connection with
	    None -> ()
	  | Some f -> at_first_connection := None; f ()))
      (*Tk_info.display "Vous tes  prsent connect"*)
    with Reconnect_later -> Tk_info.display "Impossible de se connecter"

let giveup =
  sprintf 
    "Le serveur met plus de %i %s  rpondre : inutile d'insister"
    Default.timeout sec

let timed_connect () = 
  if !dino_connection then
    Tk_info.display "Vous tes dj connect"
  else
    try
      (timed_dino_connect ~quit:false ();
       assert (on ());
       (match !at_first_connection with
	    None -> ()
	  | Some f -> at_first_connection := None; f ()))
    with 
	Reconnect_later -> Tk_info.display "Impossible de se connecter"
      | Slow_connection -> Tk_info.display giveup


      
let timeout_error = 
  sprintf "Dommage : dlai maximal de rponse dpass (%i %s)" 
    Default.timeout sec

let query_queue_is_empty () =
  Mutex.lock dqueue_mutex;
  let r = Double_queue.is_empty2 query_queue in
  Mutex.unlock dqueue_mutex;
  r

let query_queue_length () =
  Mutex.lock dqueue_mutex;
  let r = Double_queue.length2 query_queue in
  Mutex.unlock dqueue_mutex;
  r


let rec response_watcher () =
  if not (query_queue_is_empty ()) then
    (Thread.delay timeout;
     let last_response = get_last_response () in
     let now = Unix.time () in
     if now -. last_response > timeout then
       let n = query_queue_length () in
       multi_add_response (Error timeout_error) n)
  else
    Thread.delay timeout;
  
  response_watcher ()
    
let response_watcher_thread = Dodo_thread.create response_watcher ()
