(* $Id: dictionary.ml,v 1.1 2001/03/07 01:17:55 mjambon Exp $ *)


type 'a t = { mutable longer : (char * 'a t) list;
	      mutable exact : 'a option }

type 'a completion = [ `Partial of 'a list
		     | `Completed of string * 'a
		     | `Exact of 'a
		     | `Shorter of string ]

let create () = { longer = [];
		  exact = None }

let add t key data =
  let len = String.length key in
  let rec add_local t i =
    if i = len then
      t.exact <- Some data
    else
      let c = key.[i] in
      let t' = 
	try List.assoc c t.longer 
	with Not_found ->
	  let t' = create () in
	  t.longer <- (c,t') :: t.longer;
	  t' in
      add_local t' (i+1) in
  add_local t 0

let list_to_string l =
  let len = List.length l in
  let s = String.create len in
  let r = ref l in
  for i = len-1 downto 0 do
    let l = !r in
    s.[i] <- List.hd l;
    r := List.tl l
  done;
  s

let rec find_all accu (c,t) = 
  match t.exact with
      None -> List.fold_left ~f:find_all ~init:accu t.longer
    | Some data -> List.fold_left ~f:find_all ~init:(data::accu) t.longer

let rec really_complete accu = function
    { exact = None; longer = [(c,t)] } -> really_complete (c::accu) t
  | { exact = None; longer = l } -> `Partial (list_to_string accu,
					      List.fold_left 
					      ~f:find_all ~init:[] l)
  | { exact = Some data; longer = [] } -> `Completed (list_to_string accu,data)
  | { exact = Some data; longer = l } -> 
      `Partial (list_to_string accu,
		data :: List.fold_left ~f:find_all ~init:[] l)

let complete t key =
  let len = String.length key in
  let exact = ref None in
  let longer = ref [] in
  let rec find_local t i =
    if i = len then
      match t.exact with
	  None -> really_complete [] t
	| Some data ->
	    if t.longer = [] then `Exact data
	    else really_complete [] t
    else
      let c = key.[i] in
	try 
	  let t' = List.assoc c t.longer in
	  find_local t' (i+1)
	with Not_found -> `Shorter (String.sub key ~pos:0 ~len:i) in
  
  find_local t 0


open Printf

let verbose_complete t key =
  printf "Completion of \"%s\" = " key;
  let result = complete t key in
  (match result with
      `Partial l -> printf "Partial"
    | `Completed (s,_) -> printf "Completed \"%s\"" s;
    | `Exact _ -> printf "Exact"
    | `Shorter s -> printf "Shorter \"%s\"" s);
  print_newline ();
  result


