(* $Id: dino_message.ml,v 1.26 2001/06/16 17:04:57 mjambon Exp $ *)

open Printf
open Str

open Connect
open Header
open Com
open Theme
open User_info
open Dino_message_type

let prn s = Printf.printf "%s\n" s; flush Pervasives.stdout

let re_numb = regexp "|NUMB:\\(.*\\)"
and re_date = regexp "|DATE:\\(.*\\)"
and re_auth = regexp "|AUTH:\\(.*\\)"
and re_subj = regexp "|SUBJ:\\(.*\\)"
and re_them = regexp "|THEM:\\(.*\\)"
and re_cens = regexp "|CENS:\\(.*\\)"
and re_sign = regexp "|SIGN:\\(.*\\)"
and re_repl = regexp "|REPL:\\(.*\\)"
and re_effa = regexp "|EFFA:\\(.*\\)"
and re_line = regexp "||\\(.*\\)"

(*
let re_moderator = regexp "\\([^@]*\\)$"
let re_author = regexp "\\([^@]*\\)@$"
let re_god = regexp "\\([^@]*\\)@@$"
*)

let sep = regexp "[@ ]+"
let ident_cens d =
  let n = ref 0 in
  for i = 0 to String.length d - 1 do
    if d.[i] = '@' then incr n
  done;
  match !n with
      0 -> Moderator
    | 1 -> Author
    | 2 -> God
    | _ -> 
	eprintf "Champ CENS non valide (dlimiteur=%s)\n" d;
	flush stderr; Author

let split_cens s = 
  let l = bounded_full_split ~sep s 0 in
  let rec make_couples = function
      [] -> []
    | Text t::[] -> [Moderator,t]
    | Text t::Delim d::l -> (ident_cens d,t) :: make_couples l
    | _ -> eprintf "Champ CENS non valide\n"; flush stderr; [] in
  make_couples l

(*
let ident_cens s =
  let who =
    if string_match ~pat:re_moderator s ~pos:0 then
      Moderator
    else if string_match ~pat:re_author s ~pos:0 then
      Author
    else if string_match ~pat:re_god s ~pos:0 then
      God
    else failwith "Internal error (Dino_message.ident_cens)" in
  (who,matched_group 1 s)
*)  

exception First_line of string

let match_re hdr s =
  if string_match ~pat:re_numb s ~pos:0 then
    hdr.hr_num <- int_of_string (matched_group 1 s)
  else if string_match ~pat:re_date s ~pos:0 then
    hdr.hr_date <- matched_group 1 s
  else if string_match ~pat:re_auth s ~pos:0 then
    hdr.hr_author <- matched_group 1 s
  else if string_match ~pat:re_subj s ~pos:0 then
    hdr.hr_subject <- matched_group 1 s
  else if string_match ~pat:re_them s ~pos:0 then
    hdr.hr_themes <- Stringlist.of_string (matched_group 1 s)
  else if string_match ~pat:re_sign s ~pos:0 then
    hdr.hr_realauthor <- User_info.make (matched_group 1 s) "" ""
                         (* set later *)
  else if string_match ~pat:re_repl s ~pos:0 then
    hdr.hr_rep <- Intlist.of_string (matched_group 1 s)
  else if string_match ~pat:re_cens s ~pos:0 then
    let sub = matched_group 1 s in
    if sub <> "" then
      hdr.hr_cens <- split_cens sub;
  else if string_match ~pat:re_effa s ~pos:0 then
    let sub = matched_group 1 s in
    if sub <> "" then
      hdr.hr_effa <- true;
  else if string_match ~pat:re_line s ~pos:0 then
    raise (First_line (matched_group 1 s))

let get_hdr l = 
  let hdr = Header.create () in
  List.iter ~f:(match_re hdr) l;
  hdr

let cut s = 
  let len = String.length s - 2 in
  if len > 0 then String.sub s ~pos:2 ~len
  else ""

let rec set_msg msg = function
    [] -> ()
  | hd::tl ->
      try 
	(match_re msg.msg_header hd;
	 set_msg msg tl)
      with
	  First_line s -> 
	    let contents = String.concat ~sep:"\n" (s :: List.map ~f:cut tl) in
	    msg.msg_contents <- contents

	      
let get_msg l =
  let msg = create () in
  set_msg msg l;
  msg

exception Bad_number
exception Sorry

let set_author m = 
  let hdr = m.msg_header in
  match User_info.get hdr.hr_realauthor.ui_login with
      Some ui -> hdr.hr_realauthor <- ui
    | None -> ()

let really_get num =
  let msg = ref None in
  let self = Thread.self in
  let cond = Condition.create ()
  and mutex = Mutex.create () in

  let signal exn =  
    Mutex.lock mutex;
    Condition.signal cond;
    Mutex.unlock mutex in

  let action response = 
    (match response with
	 Infinite_response (message_n :: l) -> msg := Some (get_msg l)
       | Error s -> Tk_info.display s
       | other -> bad_response_light other);
    signal () in
  
  Mutex.lock mutex;
  let cmd = if !Dodo_settings.get_html then "msg2 " else "msg " in
  dino_send_command 
  [cmd;string_of_int num;"\n"] [action]
  ~error_handler:signal;
  Condition.wait cond ~locking:mutex;
  Mutex.unlock mutex;

  (match !msg with
       None -> ()
     | Some m -> set_author m);

  !msg


(*
let really_get_header num =
  let hdr = ref None in
  let action = function
      Infinite_response (message_n :: l) ->
	Tk_info.display message_n;
	hdr := Some (get_hdr l)
    | other -> bad_response other in

  dino_send_command 
  ~command:["entete ";string_of_int num;"\n"] ~actions:[action];
  
  while !hdr = None do () done;
  match !hdr with 
      Some h -> h 
    | _ -> failwith "Internal error (Dino_message.really_get_header)"
*)

(* Buffering for messages *)

let msg_tbl = Hashtbl.create 499  (* cache for messages *)
let msg_queue = Queue.create ()   (* queue : to manage the size of the cache *)

let next_tbl = Hashtbl.create 499 (* cache for messages succession *)
let prev_tbl = Hashtbl.create 499

let queue_len = ref 0
let cache_size = Default.cache_size (* maxi number of messages in the cache *)

let not_pre_read num = not (Hashtbl.mem msg_tbl num)

let not_read num = 
  try let msg = Hashtbl.find msg_tbl num in not msg.msg_read
  with Not_found -> true


let get num = 
  try 
    Some (Hashtbl.find msg_tbl num)
  with 
      Not_found ->
	let msg_opt = really_get num in
	(match msg_opt with
	     Some msg ->
	       Hashtbl.add msg_tbl ~key:num ~data:msg;
	       Queue.add num msg_queue;
	       if !queue_len > cache_size then
		 let older_num =
		   try Queue.take msg_queue 
		   with Queue.Empty -> 
		     failwith "Internal error (Dino_message.get)" in
		 Hashtbl.remove msg_tbl older_num
	       else
		incr queue_len;
	     
	   | None -> ());

	msg_opt
	      


let apply_next ~f ~th = (* asynchronously applies f to the number 
			   of the next message *)
  let action = function 
      Finite_response (1,[num_string]) ->
	f th (int_of_string num_string)
    | Error "message inexistant" ->
	Tk_info.display (Printf.sprintf "Erreur : message inexistant")
    | Error s -> Tk_info.display s
    | other -> bad_response_light other in
	
  let tn = th.theme_name and num = th.theme_last_read in
  dino_send_command 
  ["suivant ";string_of_int num;" ";tn;"\n"] [action]


let really_get_next ~tn ~num =
  let cond = Condition.create ()
  and mutex = Mutex.create () in
  let msg_num = ref None in

  let signal exn =  
    Mutex.lock mutex;
    Condition.signal cond;
    Mutex.unlock mutex in

  let action response = 
    try
      begin
	Mutex.lock mutex;
	(match response with
	     Finite_response (1,[s]) ->
	       let num' = int_of_string s in
	       if num' = 0 then
		 Tk_info.display "Pas de message suivant dans ce thme"
	       else
		 msg_num := Some num'
		   
	   | Error "message inexistant" ->
	       Tk_info.display "Pas de message suivant"
	   | Error s -> Tk_info.display s
	   | other -> bad_response_light other);
	Condition.signal cond;
	Mutex.unlock mutex
      end;
    with exn -> 
      Condition.signal cond;
      Mutex.unlock mutex; 
      raise exn in

  Mutex.lock mutex;
  dino_send_command 
  ["suivant ";string_of_int num;" ";tn;"\n"] [action]
  ~error_handler:signal;
  Condition.wait cond ~locking:mutex;

  !msg_num


let really_get_previous ~tn ~num =
  let cond = Condition.create ()
  and mutex = Mutex.create () in
  let msg_num = ref None in

  let signal exn =  
    Mutex.lock mutex;
    Condition.signal cond;
    Mutex.unlock mutex in

  let action response = 
    try
      begin
	(match response with
	     Finite_response (1,[s]) ->
	       let num' = int_of_string s in
	       if num' = 0 then
		 Tk_info.display "Pas de message prcdent dans ce thme"
	       else
		 msg_num := Some num'
		   
	   | Error "message inexistant" ->
	       Tk_info.display "Pas de message prcdent"
	   | Error s -> Tk_info.display s
	   | other -> bad_response_light other);
	Condition.signal cond
      end;
    with exn -> Condition.signal cond; raise exn in

  Mutex.lock mutex;
  dino_send_command 
  ["precedent ";string_of_int num;" ";tn;"\n"] [action] ~error_handler:signal;
  Condition.wait cond ~locking:mutex;
  
  !msg_num


                (******** High level functions ********)

let get_next ~th =
  let tn = th.theme_name
  and num = th.theme_current in
  let key = (num,tn) in
  try Some (Hashtbl.find next_tbl key)
  with Not_found ->
    let num_opt = really_get_next ~tn ~num in 
    (match num_opt with
	 None -> 
	   th.theme_read <- true
       | Some next ->
	   assert (next > 0);
	   th.theme_read <- false;
	   Hashtbl.add next_tbl ~key ~data:next;
	   Hashtbl.add prev_tbl ~key:(next,tn) ~data:num);
    num_opt

let get_previous ~th =
  let tn = th.theme_name
  and num = th.theme_current in
  let key = (num,tn) in
  try Some (Hashtbl.find prev_tbl key)
  with Not_found ->
    let num_opt = really_get_previous ~tn ~num in
    (match num_opt with
	 None -> ()
       | Some prev -> 
	   assert (prev > 0);
	   Hashtbl.add prev_tbl ~key ~data:prev;
	   Hashtbl.add next_tbl ~key:(prev,tn) ~data:num);
    num_opt


let apply_new th num_next = (* internal function *)
  if num_next = 0 then
    th.theme_read <- true
  else
    th.theme_read <- false

let check_finished = Condition.create ()
let check_mutex = Mutex.create ()

let apply_new_last th num_next = (* internal function *)
  Mutex.lock check_mutex;
  if num_next = 0 then
    th.theme_read <- true
  else
    th.theme_read <- false;
  Condition.broadcast check_finished; (* signals that the last response 
					 has been received *)
  Mutex.unlock check_mutex



let check_next th = apply_next ~f:apply_new ~th
let check_next_last th = apply_next ~f:apply_new_last ~th

let last_nbmsg = ref 0

let nbmsg () =
  let nbmsg_cond = Condition.create ()
  and nbmsg_mutex = Mutex.create () in
  let num = ref !last_nbmsg in

  let signal exn =  
    Mutex.lock nbmsg_mutex;
    Condition.signal nbmsg_cond;
    Mutex.unlock nbmsg_mutex in

  let action = function
      Finite_response (1,[s]) -> 
	let n = int_of_string s in
	Mutex.lock nbmsg_mutex;
	num := n;
	Condition.signal nbmsg_cond;
	Mutex.unlock nbmsg_mutex
    | Error s ->     
	Condition.signal nbmsg_cond;
	Tk_info.display s
    | other -> 
	Condition.signal nbmsg_cond;
	bad_response_light other in
  Mutex.lock nbmsg_mutex;
  dino_send_command ["nbmsg\n"] [action] ~error_handler:signal;
  Condition.wait nbmsg_cond ~locking:nbmsg_mutex;
  Mutex.unlock nbmsg_mutex;
  !num



let check_global () =
  let num = nbmsg () in
  if num > !last_nbmsg then
    (last_nbmsg := num;
     true)
  else false

let check_new_all () =
  if check_global () then
    begin
      if Theme.checkout () then
	(Theme.filter ();
	 let all = Theme.get_all () in
	 match all with
	     hd :: tl -> 
	       List.iter ~f:check_next tl;
	       Mutex.lock check_mutex;
	       check_next_last hd;
	       Condition.wait check_finished ~locking:check_mutex;
	       Mutex.unlock check_mutex
	   | [] -> ()) (* will not happen... *)
    end
  

let rec get_new ?(first_time=true) th =

  let curnum = th.theme_current in

  let msgopt =
    if Connect.on () || not !Dodo_settings.eco then
      
      let all = Theme.get_current_list () in
      th.theme_current <- th.theme_last_read;
      match get_next ~th with
	  (Some n) as o ->
	    if not_read n then o
	    else (th.theme_last_read <- n; 
		  th.theme_current <- n;
		  get_new th)
	| None ->
	    try
	      (assert (th.theme_read);
	       Theme.print ();
	       if Connect.succeeded () then
		 Dinorc.save ();
	       let th_new = List.find ~f:(fun th -> not th.theme_read) all in
	       assert (th != th_new);
	       (match get_next ~th:th_new with
		    (Some n) as o ->
		      if not_read n then o
		      else get_new th_new
		  | None -> get_new th_new)
	      )
	    with Not_found -> 
	      if first_time then
		(check_new_all ();
		 get_new ~first_time:false th)
	      else None
    else None in

  if msgopt = None then
    th.theme_current <- curnum;

  msgopt


exception Maximum_download

let load maxi th_init =
  let th = Theme.copy th_init in
  th.theme_current <- th.theme_last_read;
  let rec load' = function
      0 -> raise Maximum_download
    | n -> 
	match get_next ~th with
	    Some num -> 
	      th.theme_current <- num;
	      let n' = 
		if not_pre_read num then
 		  (ignore (get num); n-1)
		else n in
	      load' n'

	  | None -> n in
  load' maxi


let preload ?(maxi=Default.download_size) () =
  let l = Theme.get_current_list () in
  try
    let rest = List.fold_left ~f:load ~init:maxi l in
    Tk_info.display 
      (Printf.sprintf "%i nouveaux messages ont t tlchargs" (maxi-rest))
      
  with Maximum_download -> 
    Tk_info.display 
      (Printf.sprintf
	 "Le nombre maximal de %i messages ont t tlchargs"
	 maxi)
