(* $Id: dino_stack.ml,v 1.17 2001/06/12 23:23:18 mjambon Exp $ *)

open Tk_dino_init
open Dino_message_type
open Header

type message = Num of int | Msg of Dino_message_type.t

type t = { mutable st_string : string;
	   mutable st_msg : message }

let create num = { st_string = string_of_int num;
		   st_msg = Num num }

let list = ref []


let get_num m = 
  match m.st_msg with
      Num n -> n
    | Msg m -> m.msg_header.hr_num

let save () =
  let out_chan = open_out Default.dinostack in
  output_char out_chan '(';
  List.iter ~f:(fun m -> Printf.fprintf out_chan "%i " (get_num m)) !list;
  output_char out_chan ')';
  close_out out_chan

let format header =
 Printf.sprintf "%i : %s > %s" 
   header.hr_num header.hr_author header.hr_subject


let set_msg m = 
  match m.st_msg with
      Num n -> 
	(match Dino_message.get n with
	     Some msg ->
	       m.st_msg <- Msg msg;
	       m.st_string <- format msg.msg_header
	   | None -> ())
    | _ -> ()

let poor_set m =
  match m.st_msg with
      Num n -> m.st_string <- string_of_int n
    | _ -> ()


let is_set = ref false


let set () =
  if not !is_set then
    if Connect.on () then
      (List.iter ~f:set_msg !list;
       is_set := true)
    else
      List.iter ~f:poor_set !list

let add_msg msg =
  list := { st_string = format msg.msg_header;
	    st_msg = Msg msg } :: !list


let _ =
  Menu.add_command stack_menu ~label:"Supprimer (d)";
  Menu.add_command stack_menu ~label:"Rpondre (r)"

exception No_selection
let lb_num = function
    [] -> raise No_selection
  | `Num num::_ -> num

let see ~index =
  Listbox.see stack_lb ~index;
  Listbox.activate stack_lb ~index;
  Listbox.selection_set stack_lb ~first:index ~last:index  

let rec display_msg table _ =
  try
    let num = lb_num (Listbox.curselection stack_lb) in
    let m = table.(num) in
    let msg_num = get_num m in
    Display_dino_message.num msg_num;
    Tk_info.display (Printf.sprintf "Message %s" m.st_string);
    set_msg m;
    (*Focus.set text_sb*)
  with No_selection -> ()

and suppress_msg table _ =
  try
    let num = lb_num (Listbox.curselection stack_lb) in
    list := [];
    let maxi = Array.length table - 1 in
    for i = maxi downto num+1 do
      list := table.(i) :: !list
    done;
    for i = num-1 downto 0 do
      list := table.(i) :: !list
    done;
    display ();
    see ~index:(`Num (if num < maxi then num else num-1));
    save ()
  with No_selection -> ()

and menu_suppress_msg table num () =
  try
    (list := [];
     let maxi = Array.length table - 1 in
     for i = maxi downto num+1 do
       list := table.(i) :: !list
     done;
     for i = num-1 downto 0 do
       list := table.(i) :: !list
     done;
     display ();
     see ~index:(`Num (if num < maxi then num else num-1));
     save ())
  with No_selection -> ()

and reply_msg table _ =
  try
    let num = lb_num (Listbox.curselection stack_lb) in
    let m = table.(num) in
    match m.st_msg with
	Num n -> 
	  (match Dino_message.get n with
	       Some msg -> 
		 m.st_msg <- Msg msg; 
		 Reply.reply msg
	     | None -> ())
      | Msg msg -> Reply.reply msg

  with No_selection -> ()

and menu_reply_msg table num () =
  try
    let m = table.(num) in
    match m.st_msg with
	Num n -> 
	  (match Dino_message.get n with
	       Some msg -> 
		 m.st_msg <- Msg msg;
		 Reply.reply msg
	     | None -> ())
      | Msg msg -> Reply.reply msg
    
  with No_selection -> ()

and showmenu table _ =
    match Listbox.curselection stack_lb with
	(`Num num) as index :: _ ->
	  Menu.configure_command ~command:(menu_suppress_msg table num)
	    stack_menu (`Num 1);
	  Menu.configure_command ~command:(menu_reply_msg table num)
	    stack_menu (`Num 2);
	  Listbox.see stack_lb ~index;
	  let (x,y) = Winfo.pointerxy stack_lb in
	  Menu.popup ~x ~y stack_menu
      | _ -> ()



and bind_list table =
  Tk.bind stack_lb 
  ~events:[`KeyPressDetail "q"] ~action:(fun _ -> Tk.closeTk ());
  Tk.bind stack_lb 
  ~events:[`ButtonReleaseDetail 1] ~action:(display_msg table);
  Tk.bind stack_lb 
  ~events:[`KeyPressDetail "Return"] ~action:(display_msg table);
  Tk.bind stack_lb 
  ~events:[`ButtonPressDetail 3] ~action:(showmenu table);
  Tk.bind stack_lb 
  ~events:[`KeyPressDetail "d"] ~action:(suppress_msg table);
  Tk.bind stack_lb
  ~events:[`KeyPressDetail "r"] ~action:(reply_msg table)

and display () =
  let table = Array.of_list !list in
  bind_list table;

  let texts = List.map ~f:(fun m -> m.st_string) !list in
  Listbox.delete stack_lb ~first:(`Num 0) ~last:`End;
  Listbox.insert stack_lb ~index:`End ~texts;
  Listbox.activate stack_lb ~index:(`Num 0)

let refresh () =
  set ();
  display ()


let add () =
  add_msg !Display_dino_message.current_msg;
  display ();
  save ()


let load () = 
  try 
    let in_chan = open_in Default.dinostack in
    let lexbuf = Lexing.from_channel in_chan in
    let num_list = Lexer_dinostack.main lexbuf in
    close_in in_chan;
    list := List.map ~f:create num_list
  with _ -> ()
