(* $Id: dodo_settings.ml,v 1.17 2001/06/18 23:17:36 mjambon Exp $ *)

open Unix
open ThreadUnix
open Printf
open Str

open Dodo

type t = { dodo_label : string;
	   dodo_input : string -> unit;
	   dodo_output : unit -> string }
	   
let table = Hashtbl.create 97

let list_sep = regexp ","
let split_list s = split_delim ~sep:list_sep s

let dino_login = ref (try 
			(try getenv "DINO_LOGIN" 
			 with Not_found -> 
			   try getenv "DINO_NAME"
			   with Not_found ->
			     try getenv "OEUF_LOGIN"
			     with Not_found -> getenv "OEUF_NAME")
		      with Not_found -> "")

let dino_nick = ref (try
		       (try getenv "DINO_NICK" 
			with Not_found -> getenv "OEUF_NICK")
		     with Not_found -> !dino_login)


let dino_pass = ref (try 
		       (try getenv "DINO_PASS" 
			with Not_found -> getenv "OEUF_PASS")
		     with Not_found -> "")
		      

let dino_servers = ref [(Default.server,Default.port)]
let dino_serv = ref ("",0)

let add_server serv_port =
  let l = List.filter ~f:((<>) serv_port) !dino_servers in
  dino_servers := serv_port :: l
  
let add_server_list l = List.iter ~f:add_server (List.rev l)
			  
let re_serv = regexp "\\([^:]*\\):\\([0-9]+\\)"
let input_servers s =
  let get s accu = 
    if string_match ~pat:re_serv s ~pos:0 then
      let server = matched_group 1 s
      and port = int_of_string (matched_group 2 s) in
      (server,port) :: accu
    else
      accu in
  let l = List.fold_right ~f:get ~init:[] (split_list s) in
  add_server_list l
    

let output_servers () =
  String.concat ~sep:"," 
    (List.map ~f:(fun (serv,port) -> sprintf "%s:%i" serv port) !dino_servers)


let _ = 
  try
    let env_serv = 
      try getenv "DINO_SERV" 
      with Not_found -> getenv "OEUF_SERV" in
    let l = !dino_servers in
    let serv_port = (env_serv,Default.port) in
    if not (List.mem serv_port l) then
      dino_servers := serv_port :: l
  with Not_found -> ()


let failure_table : (string*int,int) Hashtbl.t = Hashtbl.create 13
let connection_failures serv_port =
  try Hashtbl.find failure_table serv_port with
      Not_found -> 0
let add_failure serv_port =
  let n =
    try 
      let num = Hashtbl.find failure_table serv_port in
      Hashtbl.remove failure_table serv_port;
      num 
    with
	Not_found -> 0 in
  Hashtbl.add failure_table ~key:serv_port ~data:(n+1)
  
let clear_failures () =
  Hashtbl.clear failure_table

let try_another_server () =
  match !dino_servers with
      [] -> invalid_arg "try_another_server"
    | hd::((next::tl) as l) -> dino_servers := l@[hd]; next
    | [hd] -> hd

let get_server () =
  match !dino_servers with
      [] -> invalid_arg "try_server"
    | hd::tl -> hd


let editors = ref []

let add_editor editor =
  let name = editor.ed_name in
  let l = List.filter ~f:(fun ed -> ed.ed_name <> name) !editors in
  editors := editor :: l

let check_editor editor =
  let name = editor.ed_name in
  if not (List.exists ~f:(fun ed -> ed.ed_name = name) !editors) then
    editors := editor :: !editors
  
let add_editor_list l = List.iter ~f:add_editor (List.rev l)
let check_editor_list l = List.iter ~f:check_editor (List.rev l)
	  
let re_editor = regexp "\\([NE]\\)\\(-?\\):\\([^:]*\\):\\(.*\\)"
let input_editors s =
  let rec map = function
      [] -> []
    | s::tl ->
	if string_match ~pat:re_editor s ~pos:0 then
	  let active = matched_group 2 s = "" in
	  if matched_group 1 s = "N" then
	    { ed_name = `Native;
	      ed_active = active } :: map tl
	  else
	    let edname = matched_group 3 s
	    and edcommand = matched_group 4 s in
	    { ed_name = `External (edname,edcommand);
	      ed_active = active } :: map tl
	else
	  map tl in

  let l = map (split_list s) in
  add_editor_list l;
  check_editor_list Default.editors

 
let get_editor () =
  let rec get = function
      [] -> { ed_name = `Native; ed_active = true }
    | hd::tl -> if hd.ed_active then hd else get tl in
  get !editors

let output_editors () =
  String.concat ~sep:"," 
    (List.map ~f:(function
		      { ed_name = `Native;
			ed_active = a } -> if a then "N::" else "N-::"
		    | { ed_name = `External (edname,edcommand);
			ed_active = a } -> 
			sprintf "E%s:%s:%s" 
			  (if a then "" else "-")
			  edname edcommand)
       !editors)


let _ = 
  try
    let env_editor = 
      try getenv "DINO_EDITOR" 
      with Not_found -> getenv "OEUF_EDITOR" in
    let editor = { ed_name = `External (env_editor, env_editor^" %s");
		   ed_active = true } in
    check_editor editor
  with Not_found -> ()


let dodo_eco = ref false
let eco = ref !dodo_eco
let input_eco = function
    "yes" -> dodo_eco := true; eco := true
  | _ -> dodo_eco := false; eco := false

let output_eco () =
  if !dodo_eco then "yes"
  else "no"

let dodo_anonymous = ref false
let anonymous = ref !dodo_eco
let input_anonymous = function
    "yes" -> dodo_anonymous := true; anonymous := true
  | _ -> dodo_anonymous := false; anonymous := false

let output_anonymous () =
  if !dodo_anonymous then "yes"
  else "no"


let dino_sign = ref false
let input_sign = function
    "yes" -> dino_sign := true
  | _ -> dino_sign := false
let output_sign () = 
  if !dino_sign then "yes"
  else "no"
    
let highlight_quotes = ref true
let input_hlquotes = function
    "yes" -> highlight_quotes := true
  | _ -> highlight_quotes := false
let output_hlquotes () = 
  if !highlight_quotes then "yes"
  else "no"
    
let highlight_urls = ref true
let input_hlurls = function
    "yes" -> highlight_urls := true
  | _ -> highlight_urls := false
let output_hlurls () = 
  if !highlight_urls then "yes"
  else "no"
    
let highlight_emails = ref true
let input_hlemails = function
    "yes" -> highlight_emails := true
  | _ -> highlight_emails := false
let output_hlemails () = 
  if !highlight_emails then "yes"
  else "no"

let textbg = ref (Tk_default.textbg : Tk.color)
let textfg = ref (Tk_default.textfg : Tk.color)

let textcursor : Tk.cursor ref = ref (`Xcursorfgbg ("xterm",
						    !textfg,
						    !textbg))
let textinsertbg = textfg

let bg_white = ref true
let set_white () =
  textbg := Tk_default.textbg;
  textfg := Tk_default.textfg
let set_black () =
  textbg := Tk_default.textfg;
  textfg := Tk_default.textbg

let set_color () =
  if !bg_white then set_white ()
  else set_black ();
  textcursor := `Xcursorfgbg ("xterm",
			      !textfg,
			      !textbg)


let input_bgw = function
    "yes" -> bg_white := true; set_color ()
  | _ -> bg_white := false; set_color ()

let output_bgw () = 
  if !bg_white then "yes"
  else "no"
    

let send_html = ref false

let input_send_html = function
    "yes" -> send_html := true
  | _ -> send_html := false

let output_send_html () =
  if !send_html then "yes"
  else "no"

let get_html = ref false

let input_get_html = function
    "yes" -> get_html := true
  | _ -> get_html := false

let output_get_html () =
  if !get_html then "yes"
  else "no"

let browser = ref Default.browser
let input_browser s = browser := s
let output_browser s = !browser

let mailer = ref Default.mailer
let input_mailer s = mailer := s
let output_mailer s = !mailer

let auto_unzap = ref Default.auto_unzap
let input_auto_unzap = function
    "yes" -> auto_unzap := true
  | _ -> auto_unzap := false
let output_auto_unzap () = 
  if !auto_unzap then "yes"
  else "no"

let psy = ref Default.psy
let input_psy = function
    "yes" -> psy := true
  | _ -> psy := false
let output_psy () = 
  if !psy then "yes"
  else "no"


let add t = Hashtbl.add table ~key:t.dodo_label ~data:t

let _ =
  add { dodo_label = "DINO_LOGIN";
	dodo_input = (fun s -> dino_login := s);
	dodo_output = (fun () -> !dino_login) };

  add { dodo_label = "DINO_NICK";
	dodo_input = (fun s -> dino_nick := s);
	dodo_output = (fun () -> !dino_nick) };

  add { dodo_label = "DINO_PASS";
	dodo_input = (fun s -> dino_pass := s);
	dodo_output = (fun () -> !dino_pass) };

  add { dodo_label = "DINO_SERV";
	dodo_input = input_servers;
	dodo_output = output_servers };

  add { dodo_label = "EDITOR";
	dodo_input = input_editors;
	dodo_output = output_editors };

  add { dodo_label = "ECO";
	dodo_input = input_eco;
	dodo_output = output_eco };

  add { dodo_label = "ANONYMOUS";
	dodo_input = input_anonymous;
	dodo_output = output_anonymous };

  add { dodo_label = "SIGN";
	dodo_input = input_sign;
	dodo_output = output_sign };

  add { dodo_label = "HIGHLIGHT_QUOTATIONS";
	dodo_input = input_hlquotes;
	dodo_output = output_hlquotes };

  add { dodo_label = "HIGHLIGHT_URLS";
	dodo_input = input_hlurls;
	dodo_output = output_hlurls };

  add { dodo_label = "HIGHLIGHT_EMAILS";
	dodo_input = input_hlemails;
	dodo_output = output_hlemails };

  add { dodo_label = "BG_WHITE";
	dodo_input = input_bgw;
	dodo_output = output_bgw };

  add { dodo_label = "SEND_HTML";
	dodo_input = input_send_html;
	dodo_output = output_send_html };

  add { dodo_label = "GET_HTML";
	dodo_input = input_get_html;
	dodo_output = output_get_html };

  add { dodo_label = "BROWSER";
	dodo_input = input_browser;
	dodo_output = output_browser };

  add { dodo_label = "MAILER";
	dodo_input = input_mailer;
	dodo_output = output_mailer };

  add { dodo_label = "AUTO_UNZAP";
	dodo_input = input_auto_unzap;
	dodo_output = output_auto_unzap };

  add { dodo_label = "PSY";
	dodo_input = input_psy;
	dodo_output = output_psy };
