(* $Id: reply.ml,v 1.40 2001/06/19 19:32:10 mjambon Exp $ *)

open Unix
open ThreadUnix
open Str
open Printf
open Tk

open Dodo_misc
open Dodo
open Tk_dino_init
open Header
open Dino_message_type
open New_msg
open Being_edited
open Dodo_settings

let prn s = print_string s; print_newline ()


type t = Being_edited.t

type mode = [`Read|`Edit]

let mode = ref (`Read : mode)
let current_reply = ref (None : t option)

let put_text tw buf =
  Text.delete tw ~start:beginning ~stop:text_end;
  Text.insert tw ~index:beginning ~text:buf

let suppress_last_newline text =
  let len = String.length text in (* ugly test : tk adds a newline *)
  if len > 1 then
    if text.[len-1] = '\n' && text.[len-2] = '\n' then
      String.sub ~pos:0 ~len:(len-1) text
    else text
  else text


let subst_spaces s =
  for i = 0 to String.length s - 1 do
    if s.[i] = ' ' then s.[i] <- '\160'
  done;
  s

let send rep =
  let msg = rep.rep_newmsg in
  let header = msg.msg_header in

  let nick = subst_spaces (Utf8.decode header.hr_author) in
(*
  (* Auto-record of the nickname *)
  Dodo_settings.dino_nick := nick;
*)

  let themes = List.map ~f:Utf8.decode header.hr_themes
  and replies = header.hr_rep
  and subject = Utf8.decode header.hr_subject  
  and sign = header.hr_sign
  and html = header.hr_html
  and body = Utf8.decode msg.msg_contents in
  Post.message 
    ~html
    ~themes 
    ~replies 
    ~sign 
    ~nick 
    ~subject 
    body

let cut s =
  let maxi = 50 in
  if String.length s > maxi then
    let s' = String.create maxi in
    (String.blit ~src:s ~src_pos:0 ~dst:s' ~dst_pos:0 ~len:maxi;
     for i = 1 to 3 do s'.[maxi-i] <- '.' done;
     s')
  else s

let indent msg =
  let mark = Default.reply_string in
  let text = msg.msg_contents in
  let msg_num = msg.msg_header.hr_num in
  let len = String.length text in
  if len = 0 then ""
  else
    let buf = Buffer.create ((len * 5) / 3) in
    
    let author = msg.msg_header.hr_author in
    (*Buffer.add_string buf mark;*)
    Buffer.add_string buf Default.reply_author_begin;
    Buffer.add_string buf author;
    Buffer.add_string buf Default.reply_author_middle;
    Buffer.add_string buf (string_of_int msg_num);
    Buffer.add_string buf Default.reply_author_end;
    Buffer.add_char buf '\n';
    Buffer.add_string buf Default.reply_additional_line;
    Buffer.add_char buf '\n';
    
    Buffer.add_string buf mark;
    for i = 0 to len-2 do
      match text.[i] with
	  '\n' -> Buffer.add_char buf '\n'; Buffer.add_string buf mark
	| c -> Buffer.add_char buf c
    done;
    (match text.[len-1] with
	 '\n' -> ()
       | c -> Buffer.add_char buf c);
    Buffer.add_char buf '\n';
    
    Buffer.contents buf

let save_specific reply_data tk_text =
  let msg = reply_data.rep_newmsg in
  let tk_header = reply_data.rep_hdr in
  New_msg.output_header tk_header msg.msg_header;
  let text = Text.get tk_text ~start:beginning ~stop:text_end in
  (*print_string text; flush stdout;*)
  let text' = suppress_last_newline text in
  msg.msg_contents <- text'
  

let index_current = `Mark "insert", []
let index_previous = `Mark "insert", [`Char(-1)]

let replace2 s textw _ = 
  Text.delete_char textw ~index:index_previous;
  Text.delete_char textw ~index:index_previous;
  Text.insert textw ~index:index_current ~text:s

(* Dead keys:
   dead_abovedot
   dead_abovering
   dead_acute 
   dead_belowdot 
   dead_breve
   dead_caron
   dead_circumflex 
   dead_circumflex 
   dead_diaeresis 
   dead_diaeresis 
   dead_doubleacute
   dead_grave 
   dead_macron
   dead_ogonek
*)

let bindings = [ 
  "dead_circumflex", "a", Utf8.code '';
  "dead_circumflex", "e", Utf8.code '';
  "dead_circumflex", "i", Utf8.code '';
  "dead_circumflex", "o", Utf8.code '';
  "dead_circumflex", "u", Utf8.code '';

  "dead_diaeresis", "a", Utf8.code '';
  "dead_diaeresis", "e", Utf8.code '';
  "dead_diaeresis", "i", Utf8.code '';
  "dead_diaeresis", "o", Utf8.code '';
  "dead_diaeresis", "u", Utf8.code '';
  "dead_diaeresis", "y", Utf8.code '';

  "dead_circumflex", "A", Utf8.code '';
  "dead_circumflex", "E", Utf8.code '';
  "dead_circumflex", "I", Utf8.code '';
  "dead_circumflex", "O", Utf8.code '';
  "dead_circumflex", "U", Utf8.code '';

  "dead_diaeresis", "A", Utf8.code '';
  "dead_diaeresis", "E", Utf8.code '';
  "dead_diaeresis", "I", Utf8.code '';
  "dead_diaeresis", "O", Utf8.code '';
  "dead_diaeresis", "U", Utf8.code '';
(*
  "dead_grave", "a", Utf8.code '';
  "dead_grave", "e", Utf8.code '';
  "dead_grave", "i", Utf8.code '';
  "dead_grave", "o", Utf8.code '';
  "dead_grave", "u", Utf8.code '';
  "dead_grave", "A", Utf8.code '';
  "dead_grave", "E", Utf8.code '';
  "dead_grave", "I", Utf8.code '';
  "dead_grave", "O", Utf8.code '';
  "dead_grave", "U", Utf8.code '';
				  
  "dead_acute", "a", Utf8.code '';
  "dead_acute", "e", Utf8.code '';
  "dead_acute", "i", Utf8.code '';
  "dead_acute", "o", Utf8.code '';
  "dead_acute", "u", Utf8.code '';
  "dead_acute", "y", Utf8.code '';
  "dead_acute", "c", Utf8.code '';
  "dead_acute", "A", Utf8.code '';
  "dead_acute", "E", Utf8.code '';
  "dead_acute", "I", Utf8.code '';
  "dead_acute", "O", Utf8.code '';
  "dead_acute", "U", Utf8.code '';
  "dead_acute", "Y", Utf8.code '';
  "dead_acute", "C", Utf8.code '';
*)
]

let eventlist = 
  List.rev_map
  ~f:(fun (key1, key2, subst) -> ([`KeyPressDetail key1;
				     `KeyPress;
				     `KeyReleaseDetail key2], 
				  replace2 subst))
    bindings

let bind_deadkeys textw =
  List.iter ~f:(fun (events, action) -> 
		  bind textw ~events ~action:(action textw))
    eventlist


let re_prefix = regexp "[\t >]*-?[\t ]*"
let get_lineprefix s =
  if string_match ~pat:re_prefix s ~pos:0 then
    let subs = matched_string s in
    for i = 0 to String.length subs-1 do
      if subs.[i] = '-' then subs.[i] <- ' '
    done;
    subs
  else ""

let make_ins_menu frame tw =
  let mb = Menubutton.create frame ~text:"Insrer" in
  let menu = Menu.create mb in
  Menubutton.configure mb ~menu;

  (* Special characters *)
  let menu_list = ref [] in
  let submenu_chars = ref (Menu.create mb) in
  for i = 160 to 255 do
    if i mod 10 = 0 then
      (submenu_chars := Menu.create mb;
       menu_list := (!submenu_chars,i,(min (i+9) 255))::!menu_list);
    let s = String.make 1 (char_of_int i) in
    Menu.add_command !submenu_chars
    ~label:s
    ~command:(fun () -> Text.insert tw ~index:index_current ~text:s);
  done;
  let menu_chars = Menu.create mb in
  List.iter 
  ~f:(fun (m,mini,maxi) ->
	Menu.add_cascade menu_chars ~label:(sprintf "%i-%i" mini maxi) ~menu:m)
    (List.rev !menu_list);
  Menu.add_cascade menu ~label:"caractre" ~menu:menu_chars;

  (* Files *)
  let insert_files = 
    List.iter ~f:(fun f -> 
		    let s = Editor.filecontents f in
		    Text.insert tw ~index:index_current ~text:s) in

  Menu.add_command menu
  ~label:"fichier"
  ~command:(fun () -> 
	      Fileselect.f
	      ~title:"Slection de fichiers  insrer"
	      ~action:insert_files
	      ~filter:"*"
	      ~file:"" ~multi:true ~sync:false);

  mb

let ed_action tw ed () =
  match ed.ed_name with 
      `Native -> Text.configure tw ~state:`Normal
    | `External (edname,edcommand) -> 
	let tk_contents =
	  Text.get tw ~start:beginning ~stop:text_end in
	let contents = suppress_last_newline tk_contents in
	let filename = Editor.create_file contents in
	Text.configure tw ~state:`Disabled;
	ignore (Dodo_thread.create 
		  (fun () ->
		     ignore 
		       (system 
			  (Dodo_misc.subst edcommand ~ins:filename));
		     Text.configure tw ~state:`Normal;
		     let s = Editor.filecontents filename in
		     put_text tw s) ())


let make_ed_button frame tw ed = 
  if ed.ed_active then
    let edname = Editor.get_name ed in
    let button = 
      Button.create frame ~text:edname ~takefocus:false ~pady:0
      ~command:(ed_action tw ed) in
    Some button
  else None


let rec tk_create_message ?(prefix="dition du message ") rep =
  let msg = rep.rep_newmsg in
  let hdr = msg.msg_header in
  let body = msg.msg_contents in

  let win = Toplevel.create top in
  Wm.title_set win (prefix^rep.rep_ident);
  let frame = Frame.create win in

  (* Buttons *)
  let bar = Frame.create frame in
  let button_clear = Button.create bar ~text:"Effacer"
  and button_save = Button.create bar ~text:"Sauvegarder"
  and button_send = Button.create bar ~text:"Envoyer"
  and button_later = Button.create bar ~text:"Plus tard..."
  and button_cancel = Button.create bar ~text:"Annuler" in


  let subject_frame = Frame.create frame
  and nick_frame = Frame.create frame
  and themes_frame = Frame.create frame 
  and replies_frame = Frame.create frame in

  (* Header *)
  let textwidth = 15 in
  let anchor = `W in
  let subject_label = Label.create subject_frame 
		      ~text:"Sujet" ~textwidth ~anchor
  and nick_label = Label.create nick_frame 
		   ~text:"Pseudonyme" ~textwidth ~anchor
  and themes_label = Label.create themes_frame 
		     ~text:"Thmes" ~textwidth ~anchor
  and replies_label = Label.create replies_frame
		      ~text:"En rponse " ~textwidth ~anchor in

  let width = 50 in
  let tk_header = rep.rep_hdr in
  let subject_entry = Entry.create subject_frame 
		      ~textvariable:tk_header.edit_subject ~width
  and nick_entry = Entry.create nick_frame 
		   ~textvariable:tk_header.edit_author ~width
  and themes_entry = Entry.create themes_frame 
		     ~textvariable:tk_header.edit_themes ~width
  and replies_entry = Entry.create replies_frame 
		      ~textvariable:tk_header.edit_rep ~width in  
 
  let sign_button = Checkbutton.create nick_frame 
		    ~text:(sprintf "Signer %s" !dino_login)
		    ~takefocus:false
		    ~onvalue:"y"
		    ~offvalue:""
		    ~variable:tk_header.edit_sign in


  let checkbox_html = Checkbutton.create bar 
		      ~text:"dino-HTML"
		      ~takefocus:false
		      ~onvalue:"y"
		      ~offvalue:""
		      ~variable:tk_header.edit_html in


  Textvariable.set tk_header.edit_subject hdr.hr_subject;
  Textvariable.set tk_header.edit_author hdr.hr_author;
  Textvariable.set tk_header.edit_themes 
    (String.concat ~sep:" " hdr.hr_themes);
  Textvariable.set tk_header.edit_rep (string_of_intlist hdr.hr_rep);
  Textvariable.set tk_header.edit_sign (if hdr.hr_sign then "y" else "");
  Textvariable.set tk_header.edit_html (if hdr.hr_html then "y" else "");



  let infobar = Message.create frame ~text:"" ~justify:`Left 
		~width:Tk_default.width in

  let display_info text = Message.configure infobar ~text in

  Tab_completion.bind_entry_with_themes 
  ~info:display_info themes_entry;

  let check_msg () =
    if not (Tab_completion.complete_entry_full_check
	      themes_entry Theme.completion_table)
    then 
      Some "Attention, la liste des thmes n'tait pas correcte."
    else if hdr.hr_author = "" then
      if !Dodo_settings.dino_nick = "" then
	(hdr.hr_author <- !Dodo_settings.dino_login;
	 Textvariable.set tk_header.edit_author hdr.hr_author;
	 Some "Attention, le champ  Pseudonyme  tait vide : indiquez un pseudonyme par dfaut dans la rubrique  Divers  du menu  Options ")
      else
	(hdr.hr_author <- !Dodo_settings.dino_nick;
	 Textvariable.set tk_header.edit_author hdr.hr_author;
	 Some "Attention, le champ  Pseudonyme  tait vide.")
	
    else None in


  let text_frame = Frame.create frame in
  let tk_text = Text.create text_frame ~width:80 ~state:`Normal 
		~background:!Dodo_settings.textbg
		~foreground:!Dodo_settings.textfg
		~font:Tk_default.msg_font (*~wrap:`Word*)
		~cursor:!Dodo_settings.textcursor
		~insertbackground:!Dodo_settings.textinsertbg in


  let editorsbar =
    if !editors <> [] then
      let fr = Frame.create frame in
      let ed_label = Label.create fr ~text:"diteurs : " in
      let ed_buttons = optmap ~f:(make_ed_button fr tk_text) !editors in
      pack [ed_label] ~side:`Left;
      if ed_buttons <> [] then
	(pack ed_buttons ~side:`Left;
	 Some fr)
      else None
    else None in

  let ins_mb = make_ins_menu bar tk_text in


 
(*  print_string (Text.configure_get tk_text); print_newline ();*)
  let sb = Scrollbar.create text_frame in
  Text.configure tk_text ~yscrollcommand:(Scrollbar.set sb);
  Scrollbar.configure sb ~command:(Text.yview tk_text);

  let destroy () = destroy win in

  let save () = save_specific rep tk_text in

  let clear () =
    Text.delete tk_text ~start:beginning ~stop:text_end;
    save () in
  let send () =
    save ();
    if Connect.on () || not !eco then
      match check_msg () with
	  None ->
	    send rep;
	    Being_edited.list := List.filter ~f:((!=) rep) !Being_edited.list;
	    if !Being_edited.list = [] then Being_edited.counter := 0;
	    update ();
	    destroy ()
	| Some error ->
	    display_info error
    else
      display_info 
	"Connectez-vous au pralable (vous tes en mode conomique)" in

  let later () =
    save_specific rep tk_text;
    rep.rep_not_shown <- true;
    destroy () in
  let cancel () =
    Being_edited.list := List.filter ~f:((!=) rep) !Being_edited.list;
    if !Being_edited.list = [] then Being_edited.counter := 0;
    update ();
    destroy () in


  let activate_tw () = Text.configure tk_text ~state:`Normal
  and desactivate_tw () = Text.configure tk_text ~state:`Disabled in
  activate_tw ();
  put_text tk_text body;
  if (get_editor ()).ed_name = `Native then activate_tw ()
  else desactivate_tw ();


  let check_space ?(nl=false) _ =
    let here = `Mark"insert" in
    let init = Text.index tk_text 
	       ~index:(here,[]) in
    
    let s = Text.get tk_text
	    ~start:(here, [`Linestart])
	    ~stop:(here, []) in
    
    let lineprefix = get_lineprefix s in

    let `Linechar (_,char) = init in
    let init = (init :> text_index) in
    if char > Default.line_length then
      let rec loop pos =
	let (`Linechar (line,char) as pos') = 
	  Text.search tk_text
	  ~switches:[`Backwards] ~pattern:" "
	  ~start:(pos,[`Char (-1)])
	  ~stop:(pos,[`Linestart]) in

	if char > Default.line_length then loop pos'
	else pos' in

      let pos = loop init in
      let text = 
	if nl then "\n"
	else "\n"^lineprefix in

      Text.insert tk_text ~index:(pos,[`Char 1]) ~text in
  
  let check_newline = check_space ~nl:true in

  let insert s _ = 
    Text.delete_char tk_text ~index:(`Mark"insert",[`Char(-1)]);
    Text.delete_char tk_text ~index:(`Mark"insert",[`Char(-1)]);
    Text.insert tk_text ~index:(`Mark"insert",[]) ~text:s in

  bind tk_text ~events:[`KeyPressDetail "space"] ~action:check_space;
  bind tk_text ~events:[`KeyPressDetail "Return"] ~action:check_newline;
  bind_deadkeys tk_text;

  pack [button_clear; button_save; button_send; button_later; button_cancel] 
  ~side:`Left ~anchor:`W;
  pack [checkbox_html] 
  ~side:`Left ~anchor:`W;

  pack [ins_mb] ~side:`Right ~anchor:`E;
 
  pack [subject_label] ~side:`Left;
  pack [nick_label]    ~side:`Left;
  pack [themes_label]  ~side:`Left;
  pack [replies_label] ~side:`Left;

  pack [subject_entry] ~side:`Left ~expand:true ~fill:`X;
  pack [nick_entry]    ~side:`Left ~expand:true ~fill:`X;
  pack [themes_entry]  ~side:`Left ~expand:true ~fill:`X;
  pack [replies_entry] ~side:`Left ~expand:true ~fill:`X;

  pack [sign_button]   ~side:`Left;

  bind subject_label ~events:[`Enter] 
  ~action:(fun _ -> Focus.set subject_entry);
  bind nick_label ~events:[`Enter] 
  ~action:(fun _ -> Focus.set nick_entry);
  bind sign_button ~events:[`Enter] 
  ~action:(fun _ -> Focus.set nick_entry);
  bind themes_label ~events:[`Enter] 
  ~action:(fun _ -> Focus.set themes_entry);
  bind replies_label ~events:[`Enter] 
  ~action:(fun _ -> Focus.set replies_entry);

  Button.configure button_clear ~command:clear ~takefocus:false;
  Button.configure button_save ~command:save ~takefocus:false;
  Button.configure button_send ~command:send ~takefocus:false;
  Button.configure button_later ~command:later ~takefocus:false;
  Button.configure button_cancel ~command:cancel ~takefocus:false;

  pack [tk_text] ~side:`Left ~expand:true ~fill:`Both;
  pack [sb] ~side:`Right ~expand:false ~fill:`Y;

  pack [bar;
	subject_frame;
	nick_frame;
	themes_frame;
	replies_frame] ~side:`Top ~expand:false ~fill:`X;

  (match editorsbar with
       Some fr -> pack [fr] ~side:`Top ~expand:true ~fill:`Both;
     | None -> ());
  pack [text_frame] ~side:`Top ~expand:true ~fill:`Both;
  pack [infobar] ~side:`Bottom ~expand:true ~fill:`Both;

  pack [frame] ~expand:true ~fill:`Both;
  ed_action tk_text (get_editor ()) ()
  

and show ?(prefix="dition du message ") rep =
  if rep.rep_not_shown then
    (current_reply := Some rep;
     Tk_info.display (prefix^rep.rep_ident);
     tk_create_message ~prefix rep;
     rep.rep_not_shown <- false)


and update () =
  Being_edited.async_output ();
  Menu.delete menu_replies ~first:(`Num 0) ~last:`Last;
  match !Being_edited.list with
      [] -> Pack.forget [mb_replies]
    | l ->
	List.iter 
	~f:(fun reply ->
	      Menu.add_command menu_replies
	      ~label:reply.rep_ident 
	      ~command:(fun () -> show reply)) l;
	Menubutton.configure mb_replies ~menu:menu_replies;
	pack [mb_replies] ~anchor:`W

let edit_new ?(th_list=[]) ?(msg=Dino_message_type.null) () =
  let header = msg.msg_header in
  let ident = Printf.sprintf "(%i)" (incr Being_edited.counter;
				     !Being_edited.counter) in

  let hdr = Header.edit_new header in
  let new_msg = { msg_header = hdr;
		  msg_contents = "";
		  msg_read = true (* unused *) } in

  let tk_header = New_msg.create () in
  New_msg.load_header hdr tk_header;

  hdr.hr_rep <- th_list @ hdr.hr_rep;

  let rep = { rep_ident = ident;
	      rep_hdr = tk_header;
	      rep_msg = msg;
	      rep_newmsg = new_msg;
	      rep_not_shown = true } in

  Being_edited.list := rep :: !Being_edited.list;
  update ();
  show rep



let reply msg =
  let header = msg.msg_header in
  let ident = cut (Printf.sprintf "%i : %s > %s" 
		     header.hr_num header.hr_author header.hr_subject) in

  let new_contents = indent msg in
  let hdr = Header.reply header in
  let new_msg = { msg_header = hdr;
		  msg_contents = new_contents;
		  msg_read = true (* unused *) } in

  let tk_header = New_msg.create () in
  New_msg.load_header hdr tk_header;

  let rep = { rep_ident = ident;
	      rep_hdr = tk_header;
	      rep_msg = msg;
	      rep_newmsg = new_msg;
	      rep_not_shown = true } in

  Being_edited.list := rep :: !Being_edited.list;
  update ();
  show ~prefix:"Rponse au message " rep


 
let current () =
  let msg = !Display_dino_message.current_msg in
  reply msg

let new_msg () =
  let msg = !Display_dino_message.current_msg in
  edit_new ~msg ()
