(* $Id: tag.mll,v 1.10 2001/07/06 11:34:35 mjambon Exp $ *)

{
  open Lexing
  open Tk_dino_init
   
  let zero = `Linechar (0,0)
  let absolute n = (zero,[`Char n])

  (* Quotations *)
  let quote1_tag = "q1"
  let quote2_tag = "q2"
  let quote3_tag = "q3"
  let quote4_tag = "q4"

  let url_tag = "url"
  let email_tag = "email"

  let bold_tag = "bold"
  let underline_tag = "underline"

  let color1 = `Blue
  let color2 = `Red
  let color3 = `Color "BlueViolet"
  let color4 = `Color "brown"

  let configure () =
    Text.tag_configure text ~tag:quote1_tag ~foreground:color1;
    Text.tag_configure text ~tag:quote2_tag ~foreground:color2;
    Text.tag_configure text ~tag:quote3_tag ~foreground:color3;
    Text.tag_configure text ~tag:quote4_tag ~foreground:color4;
    Text.tag_configure text ~tag:bold_tag ~font:Tk_default.msg_bold_font
    

  let set_quote1 ~start ~stop =
    Text.tag_add text ~tag:quote1_tag 
    ~start:(absolute start) ~stop:(absolute stop)
  let set_quote2 ~start ~stop =
    Text.tag_add text ~tag:quote2_tag 
    ~start:(absolute start) ~stop:(absolute stop)
  let set_quote3 ~start ~stop =
    Text.tag_add text ~tag:quote3_tag 
    ~start:(absolute start) ~stop:(absolute stop)
  let set_quote4 ~start ~stop =
    Text.tag_add text ~tag:quote4_tag 
    ~start:(absolute start) ~stop:(absolute stop)

  let set_url = 
    let counter = ref 0 in
    fun ~start ~stop url ->
      incr counter;
      let tag = url_tag^(string_of_int !counter) in
      Text.tag_add text ~tag
      ~start:(absolute start) ~stop:(absolute stop);
      Text.tag_bind text ~tag ~events:[`ButtonPressDetail 1] 
      ~action:(fun _ -> 
		 ignore 
		   (Dodo_thread.create 
		      (fun () -> 
			 ignore 
			   (Sys.command 
			      (Dodo_misc.subst 
				 !Dodo_settings.browser ~ins:url)))()));
      Text.tag_bind text ~tag ~events:[`Enter]
      ~action:(fun _ -> 
		 Text.tag_configure text ~tag 
		 ~background:(`Color "pink"));
      Text.tag_bind text ~tag ~events:[`Leave]
      ~action:(fun _ -> 
		 Text.tag_configure text ~tag 
		 ~background:!Dodo_settings.textbg);

      Text.tag_configure text ~tag ~underline:true

  let set_email = 
    let counter = ref 0 in
    fun ~start ~stop email ->
      incr counter;
      let tag = email_tag^(string_of_int !counter) in
      Text.tag_add text ~tag
      ~start:(absolute start) ~stop:(absolute stop);
      Text.tag_bind text ~tag ~events:[`ButtonPressDetail 1] 
      ~action:(fun _ -> 
		 ignore (Dodo_thread.create 
			   (fun () ->
			      ignore 
				(Sys.command 
				   (Dodo_misc.subst 
				      !Dodo_settings.mailer ~ins:email)))()));
      Text.tag_bind text ~tag ~events:[`Enter]
      ~action:(fun _ -> 
		 Text.tag_configure text ~tag 
		 ~background:(`Color "yellow"));
      Text.tag_bind text ~tag ~events:[`Leave]
      ~action:(fun _ -> 
		 Text.tag_configure text ~tag 
		 ~background:!Dodo_settings.textbg);

      Text.tag_configure text ~tag ~underline:true
}

let urlchar = [^'\000'-'\036' '\'' '(' ')' ';' '@' '<''>' '\091'-'\094' '`' '\123'-'\125' '\127'-'\255']
let urlend = [^ '.' '\000'-'\036' '\'' '(' ')' ',' ';' '@' '<''>' '\091'-'\094' '`' '\123'-'\125' '\127'-'\255']

let emailchar = ['a'-'z''A'-'Z''.''_''-''0'-'9']
let emailend = ['a'-'z''A'-'Z''_''-''0'-'9']

rule quote = parse
  | "> > > > "[^'\n']*       { let start = lexeme_start lexbuf
			       and stop = lexeme_end lexbuf in
			       set_quote1 ~start ~stop:(start+1);
			       set_quote2 ~start:(start+2) ~stop:(start+3);
			       set_quote3 ~start:(start+4) ~stop:(start+5);
			       set_quote4 ~start:(start+6) ~stop;
			       quote lexbuf }
  | "> > > "[^'\n']*         { let start = lexeme_start lexbuf
			       and stop = lexeme_end lexbuf in
			       set_quote1 ~start ~stop:(start+1);
			       set_quote2 ~start:(start+2) ~stop:(start+3);
			       set_quote3 ~start:(start+4) ~stop;
			       quote lexbuf }
  | "> > "[^'\n']*           { let start = lexeme_start lexbuf
			       and stop = lexeme_end lexbuf in
			       set_quote1 ~start ~stop:(start+1);
			       set_quote2 ~start:(start+2) ~stop;
			       quote lexbuf }
  | "> "[^'\n']*             { let start = lexeme_start lexbuf
			       and stop = lexeme_end lexbuf in
			       set_quote1 ~start ~stop;
			       quote lexbuf }
  | [^'\n']+                 { quote lexbuf }
  | '\n'                     { quote lexbuf }
  | eof                      { () }


and url = parse
  | "http"['s']?"://" urlchar* '#'? urlchar* urlend 
                              { let start = lexeme_start lexbuf
			        and stop = lexeme_end lexbuf in
				let addr = lexeme lexbuf in
				set_url ~start ~stop addr;
			        url lexbuf }
  | "ftp://" urlchar* urlend  { let start = lexeme_start lexbuf
			        and stop = lexeme_end lexbuf in
				let addr = lexeme lexbuf in
				set_url ~start ~stop addr;
				url lexbuf }
  | "www." urlchar* '#'? urlchar* urlend  
                           { let start = lexeme_start lexbuf
			     and stop = lexeme_end lexbuf in
			     let addr = lexeme lexbuf in
			     set_url ~start ~stop addr;
			     url lexbuf }
  | "ftp." urlchar* urlend  
                           { let start = lexeme_start lexbuf
			     and stop = lexeme_end lexbuf in
			     let addr = lexeme lexbuf in
			     set_url ~start ~stop addr;
			     url lexbuf }
  | eof                      { () }
  | _                        { url lexbuf }

and email = parse
  | emailchar+'@'emailchar+emailend
                            { let start = lexeme_start lexbuf
			      and stop = lexeme_end lexbuf in
			      let addr = lexeme lexbuf in
			      set_email ~start ~stop addr;
			      email lexbuf }
  | eof                      { () }
  | _                        { email lexbuf }

{
  let scan_quotes s =
    let lexbuf = Lexing.from_string s in
    quote lexbuf

  let scan_urls s =
    let lexbuf = Lexing.from_string s in
    url lexbuf

  let scan_emails s =
    let lexbuf = Lexing.from_string s in
    email lexbuf
}
