(* $Id: theme.ml,v 1.21 2001/07/06 11:34:36 mjambon Exp $ *)

open Printf
open Tk

open Dodo_settings
open Tk_dino_init
open Com
open Connect
open Theme_info

let prn s = Printf.printf "%s\n" s; flush stdout

exception Bad_name of string

type t = { theme_name : string;
	   theme_info : Theme_info.t option;
	   mutable theme_ignored : bool;
	   mutable theme_last_read : int;
	   mutable theme_current : int;
	   mutable theme_read : bool }

let create s = { theme_name = s;
		 theme_info = None;
		 theme_ignored = false;
		 theme_last_read = 0;
		 theme_current = 0;
		 theme_read = false }

let copy th = { theme_name = th.theme_name;
		theme_info = th.theme_info;
		theme_ignored = th.theme_ignored;
		theme_last_read = th.theme_last_read;
		theme_current = th.theme_current;
		theme_read = th.theme_read }


let current_theme = ref (create "dino")
let current () = !current_theme
let set t = current_theme := t

let pre_current_theme = ref (create "dino")
let pre_current () = !pre_current_theme
let pre_set t = pre_current_theme := t

let table = Hashtbl.create 149
let full_list = ref []
let list = ref []
let filter () = 
  list := List.filter ~f:(fun t -> not t.theme_ignored) !full_list

let get_list () = !list
let get_all () = !full_list

let completion_table = Dictionary.create ()
let completion_update () =
  List.iter ~f:(fun th -> Dictionary.add completion_table th.theme_name th)
    !full_list

let full = ref false (* display all themes or not ? *)

let get_current_list () = if !full then get_all () else get_list ()

let get_num = function
    [] -> 0
  | `Num num::_ -> num

let display_num = ref (fun (n : int) -> ())

let last_message th =
  let last_cond = Condition.create ()
  and last_mutex = Mutex.create () in
  let num = ref th.theme_last_read in
  let action = function
      Finite_response (1,[s]) -> 
	let n = int_of_string s in
	Mutex.lock last_mutex;
	num := n;
	Condition.signal last_cond;
	Mutex.unlock last_mutex
    | Error s -> 
	Condition.signal last_cond; 
	Tk_info.display s
    | other -> 
	Condition.signal last_cond; 
	bad_response_light other in
  Mutex.lock last_mutex;
  dino_send_command ["dernier ";th.theme_name;"\n"] [action];
  Condition.wait last_cond ~locking:last_mutex;
  Mutex.unlock last_mutex;
  !num


let string_of_theme = 
  let mo = "modr" and li = "libre" in
  let si = "sign" and an = "anonyme" in
  fun th ->
    match th.theme_info with
	None -> sprintf "Thme %s" th.theme_name
      | Some ti -> sprintf "Thme %s (%s et %s)" th.theme_name
	  (if ti.ti_signed then si else an)
	  (if ti.ti_moderated then 
	     sprintf "%s par %s" mo ti.ti_creator else li)
	  

let display_msg table _ =
  if table <> [||] then
    let num = get_num (Listbox.curselection themes_lb) in
    let t = table.(num) in
    let msg_num = t.theme_last_read in
    set t;
    (try 
       !display_num msg_num; Tk_info.display (string_of_theme t);
     with _ -> ())
(*  Focus.set text_sb*)

exception No_selection
let get_theme table =
  if table <> [||] then
    match Listbox.curselection themes_lb with
	(`Num n) as index::_ -> (table.(n),index)
      | _ -> raise No_selection
  else raise No_selection
    
let see ~index =
  Listbox.see themes_lb ~index;
  Listbox.activate themes_lb ~index;
  Listbox.selection_set themes_lb ~first:index ~last:index  

let rec print_local get =
  let l = get () in
  let table = Array.of_list l in
  bind_list table;
  let texts = List.map 
	      ~f:(fun t -> 
		    let name = 
		      let s = t.theme_name in
		      if t.theme_ignored then
			let len = String.length s in
			let s' = String.create (len+4) in
			s'.[0] <- ' ';
			s'.[1] <- ' ';
			s'.[2] <- '[';
			s'.[len+3] <- ']';
			String.blit ~src:s ~src_pos:0 ~dst:s' ~dst_pos:3 ~len;
			s'
		      else "  "^s in
		    
		    if not !eco then 
		      (if not t.theme_read then name.[0] <- 'N')
		    else
		      if Connect.off () then
			name.[0] <- '?';
		    name)
		l in
  Listbox.delete themes_lb ~first:(`Num 0) ~last:`End;
  Listbox.insert themes_lb ~index:`End ~texts;
  Listbox.activate themes_lb ~index:(`Num 0)
  
and print_all () = 
  full := true;
  print_local get_all
    
and print_selection () = 
  full := false;
  print_local get_list

and print () = if !full then print_all () else print_selection ()



and subscribe th =
  th.theme_ignored <- false;
  filter ()

and unsubscribe th =
  th.theme_ignored <- true;
  filter ()

and read_all th =
  let n = last_message th in
  th.theme_last_read <- n;
  th.theme_read <- true


and subscribe_selection table _ =
  try
    let (th,index) = get_theme table in
    if th.theme_ignored then
      (subscribe th;
       print ();
       see ~index)
  with No_selection -> ()

and unsubscribe_selection table _ =
  try
    let (th,index) = get_theme table in
    if not th.theme_ignored then
      (unsubscribe th;
       print ();
       see ~index;
       if not !full then
	 let n = match index with 
	     `Num n -> n
	   | _ -> 0 in
	 let len = Array.length table in
	 if n = len-1 then
	   (set table.(len-2);
	    let index' = `Num (len-2) in
	    Listbox.selection_set themes_lb ~first:index' ~last:index')
	 else set table.(n+1))
  with No_selection -> ()

and read_all_selection table _ =
  try
    let (th,index) = get_theme table in
    read_all th;
    print ();
    let index' = match index with
	`Num i -> if i < Array.length table - 1 then `Num (i+1) else index
      | _ -> index in
    see ~index:index'
  with No_selection -> ()


and unsubscribe_menu table th index () =
  unsubscribe th;
  print ();
  see ~index;
  try
    if not !full then
      let n = match index with 
	  `Num n -> n
	| _ -> raise No_selection in
      let len = Array.length table in
      if len > 0 then
	if n = len-1 then
	  (set table.(len-2); (* FIXME a tester *)
	   let index' = `Num (len-2) in
	   Listbox.selection_set themes_lb ~first:index' ~last:index')
	else set table.(n+1)
  with No_selection -> ()

and subscribe_menu th index () =
  subscribe th;
  print ();
  see ~index

and read_all_menu th index () =
  try
    read_all th;
    print ();
    see ~index
  with No_selection -> ()

and make_menu table th index = 
  let menu = Menu.create themes_frame in
  let len = Array.length table in
  if th.theme_ignored then
    Menu.add_command menu
    ~label:"Abonnement (a)" ~command:(subscribe_menu th index)
  else
    Menu.add_command menu
    ~label:"Dsabonnement (d)" ~command:(unsubscribe_menu table th index);

  if not th.theme_read then
    Menu.add_command menu
    ~label:"Marquer comme lu (z)" ~command:(read_all_menu th index);
  menu


and bind_menu table _ =
  match Listbox.curselection themes_lb with
      (`Num n) as index::_ ->
	Listbox.see themes_lb ~index;
	let th = table.(n) in
	let (x,y) = Winfo.pointerxy themes_lb in
	Menu.popup ~x ~y (make_menu table th index)
    | _ -> ()



and bind_list table =
  bind themes_lb 
  ~events:[`KeyPressDetail "q"] ~action:(fun _ -> closeTk ());
  bind themes_lb 
  ~events:[`ButtonReleaseDetail 1] ~action:(display_msg table);
  bind themes_lb
  ~events:[`KeyPressDetail "Return"] ~action:(display_msg table);
  bind themes_lb 
  ~events:[`ButtonPressDetail 3] ~action:(bind_menu table);
  bind themes_lb 
  ~events:[`KeyPressDetail "a"] ~action:(subscribe_selection table);
  bind themes_lb 
  ~events:[`KeyPressDetail "d"] ~action:(unsubscribe_selection table);
  bind themes_lb 
  ~events:[`KeyPressDetail "z"] ~action:(read_all_selection table)

  



let get_internal ?(ign=false) ?(last=0) s = 
  try Hashtbl.find table s
  with Not_found -> 
    let th = { theme_name = s;
	       theme_info = Theme_info.try_get s;
	       theme_ignored = ign;
	       theme_last_read = last;
	       theme_current = last;
	       theme_read = false } in
    Hashtbl.add table ~key:s ~data:th;
    th

let sort_themes = List.sort 
		  ~cmp:(fun t1 t2 -> compare t1.theme_name t2.theme_name) 

let checkout () = (* checks for new themes *)
  let l = ref [] in
  let cond = Condition.create () and mutex = Mutex.create () in

  let signal exn =  
    Mutex.lock mutex;
    Condition.signal cond;
    Mutex.unlock mutex in

  let command = ["list\n"]
  and action r = 
    Mutex.lock mutex;
    (match r with   
	 Infinite_response (header::list) ->
	   List.iter ~f:(fun s -> 
			   let len = String.length s-1 in
			   let s' =
			     if len > 0 then 
			       String.sub s ~pos:1 ~len (* FIXME *)
			     else "" in
			   l := (get_internal s')::!l) list
       | _ -> ());
    Condition.signal cond;
    Mutex.unlock mutex in

  Mutex.lock mutex;
  dino_send_command command [action] ~error_handler:signal;
  Condition.wait cond ~locking:mutex;
  let official_list = !l in
  if official_list <> [] then
    (full_list := sort_themes official_list;
     completion_update ();
     true)
  else false

let get s =
  try Hashtbl.find table s 
  with Not_found -> 
    if checkout () then (* up to date list of themes *)
      filter ();
    try Hashtbl.find table s 
    with Not_found -> 
      Tk_info.display (Printf.sprintf "Thme %s inexistant" s);
      raise (Bad_name s)

let commit user_list =
  let it_worked = checkout () in
  let default = 
    if it_worked then 
      fun th -> ()
    else
      fun th -> 
	Hashtbl.add table ~key:th.theme_name ~data:th in
 
  if not it_worked then
    (full_list := sort_themes user_list;
     completion_update ());

  let commit_loc th =
    try     
      let th' = Hashtbl.find table th.theme_name in
      th'.theme_last_read <- th.theme_last_read; 
      th'.theme_current <- th.theme_current; 
      (*	print_int th.theme_last_read;
		print_newline ();*)
      th'.theme_ignored <- th.theme_ignored;
    with Not_found -> default th in
  List.iter ~f:commit_loc user_list;
  filter ()


let last th_list =
  let cmp t1 t2 = 
    if not t2.theme_ignored then
      if t1.theme_last_read < t2.theme_last_read then t2
      else t1
    else t1 in
  
  match th_list with
      [] -> 0 (* should not happen *)
    | hd::tl -> 
	(List.fold_left ~f:cmp ~init:hd tl).theme_last_read
