(* $Id: theme_info.ml,v 1.4 2001/07/06 11:34:37 mjambon Exp $ *)

open Printf
open Str

open Com
open Connect

(*
Ok/n: Info sur le theme essais
|Le theme essais est libre, anonyme et cree par Glorrandir.  
|Il a ete cree le Mercredi 23 Septembre 1998, 10h05
#fin
Ok/n: Info sur le theme prgs
|Le theme prgs est libre, signe et cree par Zonk 1er.
|Il a ete cree le Mardi 22 Septembre 1998, 1h10
#fin
Ok/n: Info sur le theme LaTeX
|Le theme LaTeX est modere, signe et cree par Glorrandir.
|Il a ete cree le Mardi 22 Septembre 1998, 12h46
#fin
*)

type t = { ti_signed : bool;
	   ti_moderated : bool;
	   ti_creator : string;
	   ti_creation_date : string }

let null = { ti_signed = false;
	     ti_moderated = false;
	     ti_creator = "?";
	     ti_creation_date = "?" }

let str_of_bool =
  let yes = "y" and no = "n" in
  fun b -> if b then yes else no

let bool_of_str s =
  if String.length s > 0 then s.[0] = 'y'
  else false

let output_ti outchan tn ti =
  fprintf outchan "%s\n%s\n%s\n%s\n%s\n"
    tn
    (str_of_bool ti.ti_signed)
    (str_of_bool ti.ti_moderated)
    ti.ti_creator
    ti.ti_creation_date


let re_line1 = regexp "^|Le theme \\([^ ]+\\) est \\([a-z]+\\), \\([a-z]+\\) et cree par \\(.+\\)\\.$"
let re_line2 = regexp "^|Il a ete cree le \\(.+\\)$"

let parse_info line1 line2 =
  try
    if string_match ~pat:re_line1 line1 ~pos:0 then
      let ti_signed = match matched_group 3 line1 with
	  "signe" -> true
	| _ -> false in
      let ti_moderated = match matched_group 2 line1 with
	  "modere" -> true
	| _ -> false in
      let ti_creator = matched_group 4 line1 in

      let ti_creation_date =     
	if string_match ~pat:re_line2 line2 ~pos:0 then
	  matched_group 1 line2
	else raise Exit in
      
      Some { ti_signed = ti_signed;
	     ti_moderated = ti_moderated;
	     ti_creator = ti_creator;
	     ti_creation_date = ti_creation_date }
    else None

  with Exit -> None


let file = Default.theme_infos

let table = Hashtbl.create 293

let record_dodo_data name signed moderated creator date =
  let ti = { ti_signed = bool_of_str signed;
	     ti_moderated = bool_of_str moderated;
	     ti_creator = creator;
	     ti_creation_date = date } in
  Hashtbl.add table ~key:name ~data:ti


let input () = 
  try
    let inchan = open_in file in
    let rec iter () =
      try 
	let line1 = input_line inchan in
	let line2 = input_line inchan in
	let line3 = input_line inchan in
	let line4 = input_line inchan in
	let line5 = input_line inchan in
	record_dodo_data line1 line2 line3 line4 line5;
	iter ()
      with End_of_file -> close_in inchan in
    iter ()
  with _ -> ()

let really_output () =
  let outchan = open_out file in
  Hashtbl.iter ~f:(fun ~key:tn ~data:ti -> output_ti outchan tn ti)
    table;
  close_out outchan


let new_themes = ref false
let output () =
  if !new_themes then really_output ()


let really_get th_name =
  let ti = ref None in
  let self = Thread.self in
  let cond = Condition.create ()
  and mutex = Mutex.create () in

  let signal exn =  
    Mutex.lock mutex;
    Condition.signal cond;
    Mutex.unlock mutex in

  let action response = 
    (match response with
	 Infinite_response [_;line1;line2] -> 
	   new_themes := true;
	   ti := parse_info line1 line2
       | Error s -> ()
       | other -> ());
    Mutex.lock mutex;
    Condition.signal cond;
    Mutex.unlock mutex in
  
  Mutex.lock mutex;
  dino_send_command ["info ";th_name;"\n"] [action] ~error_handler:signal;
  Condition.wait cond ~locking:mutex;
  Mutex.unlock mutex;

  !ti

let get th_name =
  try Some (Hashtbl.find table th_name)
  with Not_found -> 
    let result = really_get th_name in
    (match result with
	 None -> ()
       | Some ti -> Hashtbl.add table ~key:th_name ~data:ti);
    result

let try_get th_name =
  try Some (Hashtbl.find table th_name)
  with Not_found -> None

let clean () = Unix.unlink file
