
/************************************************************************/
/* commande 'sall', par Sbastien Lacour (DSM97), mai 2000              */
/************************************************************************/

/* POSIX thread Safe (with exceptions ...) */
/* exceptions : because of 'regcomp', 'reg...' */

#include "sall.h"


/************************************************************************/
/* void *allocate_mem (size_t sz)                                       */
/* alloue de la mmoire, et renvoie le pointeur vers la zone de mmoire */
/* rserve, puis si le pointeur renvoy par malloc est NULL, il n'y a  */
/* plus de mmoire ==> message d'erreur et quitter le prgrm             */
/************************************************************************/

void *allocate_mem (size_t sz)
{
   void *ptr;

   LOG_IN();
   ptr = (void*)malloc(sz);
   if ( ptr == NULL )
   {
      perror("\a(malloc) not enough memory ");
      LOG_OUT();
      exit(8);
   }
   LOG_OUT();
   return(ptr);
}


/************************************************************************/
/* void init_stack (already_type **stck)                                */
/* initialise la pile des logins dj rencontrs (->pile vide)          */
/************************************************************************/

void init_stack (already_type **stck)
{
   LOG_IN();
   *stck = NULL;
   LOG_OUT();
}


/************************************************************************/
/* void empty_stack (already_type **begin)                              */
/* vide la pile des logins dj rencontrs et libre la mmoire         */
/************************************************************************/

void empty_stack (already_type **begin)
{
   already_type *aux;
   LOG_IN();
   while (*begin!=NULL)
   {
      aux = (*begin)->next;
      free(*begin);
      *begin = aux;
   }
   *begin = NULL;
   LOG_OUT();
}


/************************************************************************/
/* int in_stack (char *s,already_type *begin)                           */
/* retourne 0 si la chaine 's' ne se trouve pas dans la pile, et 1 si   */
/* cette chaine se trouve dans la pile des logins                       */
/************************************************************************/

int in_stack (char *s,already_type *begin)
{
   int already;
   already_type *aux;

   LOG_IN();
   already = 0;
   aux = begin;
   while ( aux != NULL )
   {
      if ( !strcmp(aux->login,s) ) already = 1;
      aux = aux->next;
   }
   LOG_OUT();
   return(already);
}


/************************************************************************/
/* void add_stack (char *s,already_type **begin)                        */
/* ajouter le login 's'  la pile des logins dj rencontrs            */
/************************************************************************/

void add_stack (char *s,already_type **begin)
{
   already_type *end,*alrdy;

   LOG_IN();
   end = *begin;
   if ( end != NULL )
      while ( end->next != NULL ) end = end->next;
   alrdy = (already_type*)allocate_mem(sizeof(already_type));
   if ( *begin == NULL ) *begin = end = alrdy;
   else
   {
      end->next = alrdy;
      end = alrdy;
   }
   end->next = NULL;
   strcpy(end->login,s);
   LOG_OUT();
}


/************************************************************************/
/* void rcv_lgn (int fd,char *)                                         */
/* recoit du file descriptor une ligne NULL-terminated                  */
/************************************************************************/

void rcv_lgn (int fd,char *s)
{
   char c;

   LOG_IN();

   strcpy(s,"");
   do
   {
/* attente active ?????????????? */
      while ( 0 == read(fd,&c,sizeof(char)) ) ;
      strncat(s,&c,1);
   } while ( c!='\n' && strlen(s)<MAX_SIZE );
   s[strlen(s)-1] = '\0';
   LOG_OUT();
   return;
}


/************************************************************************/
/* void clnup_handler (void *ptr)                                       */
/* fonction de nettoyage apres l'ANNULATION (cancel) d'un thread,       */
/* mais non apres une sortie normale avec exit !!!                      */
/************************************************************************/

void clnup_handler (void *ptr)
{
   interest_t *aux_host;

   LOG_IN();
   aux_host = (interest_t*)ptr;

   aux_host->info = (info_t*)allocate_mem(sizeof(info_t));
   aux_host->info->next = NULL;
   strcpy(aux_host->info->user_logged,"no_reply");
   strcpy(aux_host->info->log_time,"");
   strcpy(aux_host->info->host_name,"");
   strcpy(aux_host->info->full_name,"*** no reply ***");
   strcpy(aux_host->info->group,"");
   strcpy(aux_host->info->console,"console");
   strcpy(aux_host->info->nickname,"");

   if ( pthread_detach(pthread_self()) )
   {
      perror("\apthread_detach ");
      LOG_OUT();
      exit(15);
   }
   LOG_OUT();
   return;
}


/************************************************************************/
/* int get_daemon (char *h)                                             */
/* connecte la socket sur le daemon de 'sall' ; la valeur de            */
/* retour est le descripteur de fichier ou -1 en cas d'erreur           */
/************************************************************************/

int get_daemon (char *h)
{
   int x,file_descr,flg;
   struct sockaddr_in addr_srv;
   int length,err;
   char *s;
   struct hostent *server_in;
   struct hostent *server_out = NULL;
   void *buffer;
   int sz = 700; /* cette valeur marche pour tous les cas tests. */
                 /* Aucune page de manuel ne donne la valeur  */
                 /* rserver pour le buffer */
   LOG_IN();
   server_in = (struct hostent*)allocate_mem(sz);
   buffer = (void*)allocate_mem(sz);
   s = (char*)allocate_mem((MAX_SIZE+1)*sizeof(char));
   length = sizeof(struct sockaddr_in);
   addr_srv.sin_family = AF_INET;
   addr_srv.sin_port = htons(IP_PORT);
#ifdef LINUX
   x = gethostbyname_r(h,server_in,(char*)buffer,sz,&server_out,&err);
   if ( x )
   {
      perror("\aGETHOSTBYNAME ");
      return(-1);
   }
#endif
#ifdef SOLARIS
   server_out = gethostbyname_r(h,server_in,(char*)buffer,sz,&err);
#endif
   if (server_out == NULL)
   {
      perror("\aGETHOSTBYNAME ");
      return(-1);
   }
   memcpy(&(addr_srv.sin_addr.s_addr),server_in->h_addr_list[0],
          sizeof(addr_srv.sin_addr.s_addr));
   free(server_in);
   free(buffer);
   /* se connecter sur le port IP_PORT et */
   /* vrifier que c'est bien le serveur de sall avec la */
   /* chaine de caractere initiale MAGIC_STRING, sinon, on */
   /* on abandonne, et on dclare "no reply from host" */

//   pthread_testcancel();

   file_descr = communic(0);
   if (file_descr == -1)
   {
      perror("\aCOMMUNIC ");
      LOG_OUT();
      return(-1);
   }
   flg = fcntl(file_descr,F_GETFD);
   if (flg<0) { perror("\afcntl "); exit(13); }
   if ( -1 == fcntl(file_descr,F_SETFD,flg|O_NONBLOCK) )
   {
      perror("\aFCNTL ");
      LOG_OUT();
      return(-1);
   }
   if (pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL))
      { perror("\apthread_setcancelstate "); exit(18); }
   x = connect(file_descr,(struct sockaddr *)&addr_srv,length);
   if (pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL))
      { perror("\apthread_setcancelstate "); exit(19); }
   if ( x )
   {
      close(file_descr);
      return(-1);
   }
   rcv_lgn(file_descr,s);
   if ( strcmp(s,MAGIC_STRING) )
   {
      shutdown(file_descr,2);
      close(file_descr);
      pthread_cancel(pthread_self());
   }

   free(s);
   LOG_OUT();
   return(file_descr);
}


/************************************************************************/
/* void set_them_free (void)                                            */
/* fait les free qu'il faut sur les *hosts[], mach_indx, pthrid         */
/************************************************************************/

void set_them_free ()
{
   info_t *next_info, *aux_info;
   interest_t *next_host, *aux_host;

   LOG_IN();
   aux_host = hosts;
   while ( aux_host != NULL )
   {
      aux_info = aux_host->info;
      next_host = aux_host->next;
      while ( aux_info != NULL )
      {
         next_info = aux_info->next;
         free(aux_info);
         aux_info = next_info;
      }
      aux_host = next_host;
   }

   free(hosts);
   LOG_OUT();
}


/************************************************************************/
/* int recognize (char *, char *)                                       */
/* retourne 1 si le resolveur de noms DNS connait la machine obtenue    */
/* par concatenation des deux chaines passees en arguments, sinon       */
/* retourne 0                                                           */
/************************************************************************/

int recognize (char *part1, char *part2)
{
#ifdef LINUX
   int x;
#endif
   int ln;
   char *fullhostname;
   struct hostent *hostentry, *result = NULL;
   int sz = 700; /* cette valeur marche pour tous les cas tests. */
                 /* Aucune page de manuel ne donne la valeur  */
                 /* rserver pour le buffer */
   int err;
   void *buffer;

   buffer = (void*)allocate_mem(sz);
   hostentry = (struct hostent*)allocate_mem(sizeof(struct hostent));
   ln = 2 + strlen(part1) + strlen(part2);
   fullhostname = (char*)allocate_mem(ln*sizeof(char));

   sprintf(fullhostname, "%s%s", part1, part2);
#ifdef SOLARIS
   result = gethostbyname_r(fullhostname,hostentry,
                               (char*)buffer,sz,&err);
#endif
#ifdef LINUX
   x = gethostbyname_r(fullhostname, hostentry, (char*)buffer, sz,
                       &result, &err);
#ifdef pasfounon
        /* la verification de result == NULL suffit, car gethostbyname_r n'est */
        /* pas documente sous linux, et la valeur de retour peut etre autre chose */
        /* que 0 (par exemple 2) dans certains cas qui ne sont pas des erreurs... */
   if ( x )
   {
      perror("\agethostbyname_r");
      return(0);
   }
#endif   /* pasfounon */
#endif

   free(fullhostname);
   free(buffer);
   if ( result == NULL )
   {
      free(hostentry);
      return(0);
   }
   else
   {
      free(hostentry);
      return(1);
   }
}


/************************************************************************/
/* void determine_host (char **name_out, char *name_in)                 */
/* essaie de determiner la machine passee dans 'choice.argmt.host',     */
/* soit une adresse IP, soit un nom de machine (auquel on essaie        */
/* d'ajouter differents noms de domaine), et si toutes les tentatives   */
/* echouent, terminer le programme en disant qu'on n'a pas trouve de    */
/* telle machine                                                        */
/************************************************************************/

void determine_host (char **name_out, char *name_in)
{
   in_addr_t addr;
#ifdef LINUX
   int x;
#endif
   LOG_IN();


//if ( (addr = xxx+xxx*256+xxx*256*256+xxx*256*256*256) != (in_addr_t)-1 )
/* cette fonction "inet_addr" n'est pas reentrante, et je n'en ai pas */
/* trouve de version reentrante ... attention au multi-thread ... */
   if ( (addr = inet_addr(name_in) ) != (in_addr_t)-1 )
   {                 /* on a reconnu une adresse IP */
      struct hostent *hostentry, *result = NULL;
      int sz = 700; /* cette valeur marche pour tous les cas tests. */
                    /* Aucune page de manuel ne donne la valeur  */
                    /* rserver pour le buffer */
      int err;
      void *buffer;

      buffer = (void*)allocate_mem(sz);
      hostentry = (struct hostent*)allocate_mem(sizeof(struct hostent));
#ifdef SOLARIS
      result = gethostbyaddr_r((void*)&addr, sizeof(addr), AF_INET,
                               hostentry, (char*)buffer, sz, &err);
#endif
#ifdef LINUX
      x = gethostbyaddr_r((void*)&addr, sizeof(addr), AF_INET,
                               hostentry, (char*)buffer, sz, &result, &err);
      if ( x )
      {
         perror("\agethostbyaddr_r");
         return;
      }
#endif
      *name_out = (char*)allocate_mem( sizeof(char) * ( 2 +
                                   strlen(hostentry->h_name) ) );
      strcpy(*name_out, hostentry->h_name);
      free(buffer);
      free(hostentry);
   }
   else  /* on n'a pas reconnu d'adresse IP et le nom de la machine */
   {     /* n'est entre en dur dans le code source */
      char *fullhostname;
      int ln;

      ln = strlen(name_in) + 2 + max(
           strlen(RESIDENCE_DOMAIN_NAME),strlen(DOMAIN_NAME));
      fullhostname = (char*)allocate_mem(ln*sizeof(char));

      if ( recognize(name_in, "") )
         sprintf(fullhostname, "%s", name_in);
      else
         if ( recognize(name_in, DOMAIN_NAME) )
            sprintf(fullhostname, "%s%s", name_in,
                                          DOMAIN_NAME);
         else
            if ( recognize(name_in, RESIDENCE_DOMAIN_NAME) )
               sprintf(fullhostname, "%s%s", name_in,
                                          RESIDENCE_DOMAIN_NAME);
            else
            {
               fprintf(stderr,"\a%s : I do not know that host ...\n",
                              name_in);
               free(fullhostname);
               set_them_free();
               exit(15);
            }
      *name_out = (char*)allocate_mem( sizeof(char) * ln );
      strcpy(*name_out, fullhostname);
      free(fullhostname);
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* version_t find_version (char *s)                                     */
/* renvoie le numero de verion du daemon salld, ou bien NO_VERSION si   */
/* aucun numero de version n'est reconnu                                */
/************************************************************************/

version_t find_version (char *s)
{
   if ( !strcmp(s, CURRENT_VERSION) ) return CURRENT;

   if ( !strcmp(s, "salld_1.1") ) return VERSION_1_1;

   return NO_VERSION;
}


/************************************************************************/
/* void *launch_request (void*)                                         */
/* lance la requete (POSIX thread) des logs sur une machine dont le     */
/* numro est pass en argument                                         */
/************************************************************************/

void *launch_request (void *ptr)
{ /* POSIX threads ... */
   int file_descr;
   char *full_host_name;
   char s[MAX_SIZE];
   interest_t *aux_host;
   info_t *info;
   version_t version;

   LOG_IN();
   info = NULL;
   aux_host = (interest_t*)ptr;
   if (pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED,NULL))
      { perror("\apthread_setcanceltype "); exit(16); }
   if (pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL))
      { perror("\apthread_setcancelstate "); exit(17); }
   pthread_cleanup_push(clnup_handler,(void*)aux_host);

   determine_host(&full_host_name, aux_host->name);
   file_descr = get_daemon(full_host_name);

   if (file_descr == -1)
      pthread_cancel(pthread_self());

      /* recevoir le numero de version du daemon salld */
   rcv_lgn(file_descr, s);
      /* si cette ligne lue NE correspond PAS a un numero de version */
      /* connu, alors on place version = NO_VERSION et on arrete de */
      /* de dialoguer avec ce demon que l'on ne reconnait pas */
      /* sinon on lit la ligne suivante qui sera la */
      /* premiere ligne de donnees */
   version = find_version(s);
   if ( version != NO_VERSION ) rcv_lgn(file_descr, s);
   else
   {
      fprintf(stderr, "\a*** You should upgrade your version of sall-client! ***\n");
      fprintf(stderr,   "*** or the version of the salld daemon running on   ***\n");
      fprintf(stderr,   "*** %-47s ***\n", full_host_name);
      LOG_OUT();
      pthread_cancel(pthread_self());
   }
   while ( strcmp(s,MAGIC_STRING) )
   {
      if ( info == NULL )
          info = aux_host->info = (info_t*)allocate_mem(sizeof(info_t));
      else
      {
         info->next = (info_t*)allocate_mem(sizeof(info_t));
         info = info->next;
      }
      info->next = NULL;
      switch (version)
      {
         case CURRENT:
            strcpy(info->user_logged,s);
            rcv_lgn(file_descr,info->log_time);
            rcv_lgn(file_descr,info->full_name);
            rcv_lgn(file_descr,info->console);
            rcv_lgn(file_descr,info->group);
            rcv_lgn(file_descr,info->host_name);
            rcv_lgn(file_descr,info->nickname);
            break;
         case VERSION_1_1:  /* la version 1.1 donne les champs dans le meme ordre */
         case NO_VERSION:   /* que la version du demon qui ne declarait pas de version */
            strcpy(info->user_logged,s);
            rcv_lgn(file_descr,info->log_time);
            rcv_lgn(file_descr,info->host_name);
            rcv_lgn(file_descr,info->full_name);
            rcv_lgn(file_descr,info->group);
            rcv_lgn(file_descr,info->console);
            rcv_lgn(file_descr,info->nickname);
            break;
         default:
            fprintf(stderr, "\aWarning: unrecognized daemon running on %s!\n",
                            aux_host->name);
            aux_host->info->next = NULL;
            strcpy(aux_host->info->user_logged,"no_reply");
            strcpy(aux_host->info->log_time,"");
            strcpy(aux_host->info->host_name,"");
            strcpy(aux_host->info->full_name,"*** no reply ***");
            strcpy(aux_host->info->group,"");
            strcpy(aux_host->info->console,"console");
            strcpy(aux_host->info->nickname,"");
            break;
      }
      rcv_lgn(file_descr, s);
   };

   close(file_descr);
   pthread_cleanup_pop(0);
   pthread_exit(NULL);
   LOG_OUT();
   return(NULL);
}


/************************************************************************/
/* void display_remote (info_t *, char *)                               */
/* affiche les lignes des remote login faits sur la machine 'mach'      */
/************************************************************************/

void display_remote (info_t *info, char *name)
{
   info_t *aux_info;
   char to_print[MAX_SIZE+1];

   LOG_IN();
   aux_info = info;

   while ( aux_info != NULL )
   {
      if ( !strcmp(aux_info->console,"remote-locked") ||
           !strcmp(aux_info->console,"remote") )
      {
         snprintf(to_print,MAX_SIZE,"%s@%s ---> %s",aux_info->user_logged,
                aux_info->host_name,name);
         if ( !strcmp(aux_info->console,"remote-locked") )
            strncat(to_print," [locked]",MAX_SIZE-strlen(to_print));
         printf("%s\n",to_print);
      }
      aux_info = aux_info->next;
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void *display_line (void *ptr)                                       */
/* affiche la ligne concernant une machine interrogee                   */
/************************************************************************/

void *display_line (void *ptr)
{
   interest_t *aux_host;
   info_t *info;
   int remote;
   already_type *stck;
   char to_print[MAX_SIZE+1],aux[MAX_SIZE+1];

   LOG_IN();
   init_stack(&stck);
   aux_host = (interest_t*)ptr;

   pthread_join(aux_host->pthrid,NULL);
   aux_host->pthrid = main_pthread_id;
   info = aux_host->info;

   /* remplacer la mise en place de la semaphore par une criture */
   /* en une seule fois de toute la ligne, qui est stocke dans */
   /* la chaine 'to_print' ... le printf est reentrant : OK */

   if (info != NULL) /* il y des gens loggus sur cette machine */
   {
      remote = 0;
      snprintf(to_print,MAX_SIZE,"%13.13s :",aux_host->name);
      while ( info != NULL )
      {
         int already;

         already = in_stack(info->user_logged,stck);

         if ( !strcmp(info->console,"remote-locked") ||
              !strcmp(info->console,"remote") )
            remote = 1;
         if (!strcmp(info->user_logged,"no_reply"))
         {
            snprintf(aux,MAX_SIZE,"\033[0m \033[4mno reply from host\033[0m");
            strncat(to_print,aux,MAX_SIZE-strlen(to_print));
         }
         else
            if ( choice.request&LONG_FORMAT || !already )
            {
               if (!strcmp("console",info->console) ||
                   !strcmp("locked",info->console))
               {
                  snprintf(aux,MAX_SIZE,"\033[1m");
                  strncat(to_print,aux,MAX_SIZE-strlen(to_print));
               }
               if (!strcmp("locked",info->console))
               {
                  snprintf(aux,MAX_SIZE," [%.8s]\033[0m",info->user_logged);
                  strncat(to_print,aux,MAX_SIZE-strlen(to_print));
               }
               else
               {
                  snprintf(aux,MAX_SIZE," %.8s\033[0m",info->user_logged);
                  strncat(to_print,aux,MAX_SIZE-strlen(to_print));
               }
               if (!already)
               {
                  if (choice.request&NICKNAME || choice.request&GROUPS)
                  {
                     snprintf(aux,MAX_SIZE,"(");
                     strncat(to_print,aux,MAX_SIZE-strlen(to_print));
                  }
                  if (choice.request&NICKNAME)
                  {
                     snprintf(aux,MAX_SIZE,"%.11s",info->nickname);
                     strncat(to_print,aux,MAX_SIZE-strlen(to_print));
                  }
                  if (choice.request&NICKNAME && choice.request&GROUPS)
                  {
                     snprintf(aux,MAX_SIZE,",");
                     strncat(to_print,aux,MAX_SIZE-strlen(to_print));
                  }
                  if (choice.request&GROUPS)
                  {
                     snprintf(aux,MAX_SIZE,"%s",info->group);
                     strncat(to_print,aux,MAX_SIZE-strlen(to_print));
                  }
                  if (choice.request&NICKNAME || choice.request&GROUPS)
                  {
                     snprintf(aux,MAX_SIZE,")");
                     strncat(to_print,aux,MAX_SIZE-strlen(to_print));
                  }
               }
            }
         if (!already) add_stack(info->user_logged,&stck);
         info = info->next;
      }
      printf("%s\n",to_print);
      if (remote) display_remote(aux_host->info, aux_host->name);
   } /* il y avait des gens loggus sur cette machine */
   empty_stack(&stck);

   LOG_OUT();
   pthread_exit(NULL);
   return(NULL);
}


/************************************************************************/
/* void query_host (interest_t *aux_host)                               */
/* lance la requete de 'sall' sur la machine an argument                */
/************************************************************************/

void query_host (interest_t *aux_host)
{ /* POSIX threads ... */
   pthread_attr_t attrb;

   LOG_IN();
   if (pthread_attr_init(&attrb))
   {
      perror("\apthread_attr_init ");
      exit(11);
   }
   if ( pthread_create(&(aux_host->pthrid),&attrb,launch_request,(void*)aux_host) )
      perror("\apthread_create ");
   if ( choice.request&LIST )
      if ( pthread_create(&(aux_host->dsp_thrd),&attrb,display_line,(void*)aux_host) )
         perror("\apthread_create ");
   if (pthread_attr_destroy(&attrb))
   {
      perror("\apthread_attr_destroy ");
      exit(13);
   }
   LOG_OUT();
   return;
}


/************************************************************************/
/* void query (void)                                                    */
/* lance la requete de 'sall' sur les machines qui doivent etre         */
/* interrogees                                                          */
/************************************************************************/

void query ()
{
   interest_t *current;

   LOG_IN();

   current = hosts;
   while ( current != NULL )
   {
      query_host (current);
      current = current->next;
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void get_index_machines (rooms)                                      */
/* stocke dans la liste chainee 'hosts' les noms des hotes a interroger */
/* et lance les threads pour les requetes                               */
/************************************************************************/

void get_index_machines (rooms rm)
{
   int i=-1;
   rooms r=-1;
   interest_t *current = NULL;

   LOG_IN();

   while ( (int)r <= (int)rm )
   {
      i++;
      if ( !strcmp(machines[i], "") ) r++;
      else
      {
         if ( ( ( rm == r ) && ( rm < ALL ) ) ||
              ( ( rm >= ALL ) && ( r < rm ) ) )
         {                            /* la machine [i] est a interroger */
            if ( current == NULL )
               current = hosts = (interest_t*)allocate_mem(sizeof(interest_t));
            else
            {
               current->next = (interest_t*)allocate_mem(sizeof(interest_t));
               current = current->next;
            }
            current->name = machines[i];
            current->info = NULL;
            current->next = NULL;
            current->pthrid = current->dsp_thrd = main_pthread_id;
         }
      }
   }

   query();

   LOG_OUT();
   return;
}


/************************************************************************/
/* void wait_for_pthrds ()                                              */
/* attend bien sagement la terminaison des threads POSIX launch_request */
/************************************************************************/

void wait_for_pthrds ()
{
   interest_t *current;

   LOG_IN();

   current = hosts;
   while ( current != NULL )
   {
      if ( current->pthrid != main_pthread_id )
      {
         pthread_join(current->pthrid,NULL);
         current->pthrid = main_pthread_id;
      }
      current = current->next;
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void impatience (int sig)                                            */
/* annule les threads qui pataugent, apres MAX_TIME secondes coules   */
/************************************************************************/

void impatience (int sig)
{
   interest_t *aux_host;

   LOG_IN();
   aux_host = hosts;
/* r-enclencher l'alarme ... je crois bien que c'est un peu cochon */
/* mais a semble necessaire, quand le signal est capt au cours de */
/* l'excution d'un 'connect' ... seulement sous SOLARIS */
#ifdef SOLARIS
   alarm(1);
#endif

   while ( aux_host != NULL )
   {
      if ( !pthread_equal(aux_host->pthrid, main_pthread_id) &&
           !pthread_equal(aux_host->pthrid, pthread_self()) )
         pthread_cancel(aux_host->pthrid);
      aux_host = aux_host->next;
   }
   if (!pthread_equal(main_pthread_id,pthread_self()))
     pthread_cancel(pthread_self());

   LOG_OUT();
   return;
}


/************************************************************************/
/* void launch_timer (void)                                             */
/* lance le compte  rebours pour eviter d'attendre trop longtemps      */
/* (voire indfiniment) si une machine rpond trop lentement (ou pas    */
/* du tout !)                                                           */
/************************************************************************/

void launch_timer ()
{
   struct sigaction sa;

   LOG_IN();

   alarm((unsigned int)MAX_TIME);

   sa.sa_handler = impatience;
   sigemptyset(&(sa.sa_mask));
/* option sa_flags incertaine ... */
   sa.sa_flags = 0;
   if ( sigaction(SIGALRM,&sa,NULL) )
   {
      perror("\aSIGACTION ");
      LOG_OUT();
      exit(9);
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void usage (void)                                                    */
/* affiche le message d'erreur sur la sortie d'erreur, le mode d'emploi */
/* et quitte le prgrm                                                   */
/************************************************************************/

void usage (char *prgrm)
{
   LOG_IN();
   fprintf(stderr,"\aUSAGE :\n%s [-h host [-ls]] | [-n pattern [-l]]\n",prgrm);
   fprintf(stderr,"%s [-afgl] | [ 1 | 12 | 2 | 6 | 9 ]\n",prgrm);
   fprintf(stderr,"       -a : print all machines\n");
   fprintf(stderr,"       -f : print nicknames\n");
   fprintf(stderr,"       -g : print groups\n");
   fprintf(stderr,"  -h host : give all info on the host\n");
   fprintf(stderr,"       -l : print all sessions\n");
   fprintf(stderr,"-n regexp : grep part of login\n");
   fprintf(stderr,"       -s : silent (do not print details on sessions)\n");
   fprintf(stderr,"   1 | 12 : \042les pices\042\t");
   fprintf(stderr,"6 : \042les vents\042\n");
#ifdef lugdunum
   fprintf(stderr,"6 : \042lugdunum\042\n");
#endif
   fprintf(stderr,"        2 : \042les les\042\t");
   fprintf(stderr,"        9 : \042les dessins anims\042\n");
   LOG_OUT();
   exit(1);
}


/************************************************************************/
/* void get_option (char *prgrm,char *argmt,expect_type *expct)         */
/* lit la ou les options donnee(s) en argument 2, et complete choice,   */
/* et positionne la nature du prochain argument attendu 'expct'         */
/************************************************************************/

void get_option (char *prgrm,char *argmt,expect_type *expct)
{
   int i,l;

   LOG_IN();
   l = strlen(argmt);
   for (i=1 ; i<l ; i++)
      switch (argmt[i])
      {
         case 'g' : choice.request|=GROUPS;
                    break;
         case 'f' : choice.request|=NICKNAME;
                    break;
         case 'l' : choice.request|=LONG_FORMAT;
                    break;
         case 'a' : choice.request|=ALL_MACHINES;
                    break;
         case 's' : choice.request|=SILENT;
                    break;
         case 'h' : choice.request = choice.request & (~LIST);
                    choice.request |= HOST;
                    *expct=EXPECT_HOST_NAME;
                    break;
         case 'n' : choice.request = choice.request & (~LIST);
                    choice.request |= LOGIN;
                    *expct=EXPECT_LOGIN;
                    break;
         default : usage(prgrm);
      }
   LOG_OUT();
}


/************************************************************************/
/* void get_argmt (char *prgrm,char *argmt,expect_type *expct)          */
/* lit l'argument pass en parametre 2, et complete choice si aucune    */
/* erreur n'est dtecte, et positionne 'expct'                         */
/************************************************************************/

void get_argmt (char *prgrm,char *argmt,expect_type *expct)
{
   LOG_IN();
   switch (*expct)
   {
      case EXPECT_FLAG|EXPECT_ROOM_NBR : case EXPECT_ROOM_NBR :
         if (choice.request==LIST) choice.request=DRAW_ROOM;
         else usage(prgrm);
         if ( 0==strcmp(argmt,"1") || 0==strcmp(argmt,"12") )
            choice.argmt.rm = EPICES;
         if ( 0==strcmp(argmt,"2") ) choice.argmt.rm = ILES;
         if ( 0==strcmp(argmt,"6") ) choice.argmt.rm = VENTS;
#ifdef lugdunum
         if ( 0==strcmp(argmt,"6") ) choice.argmt.rm = LUGDUNUM;
#endif
         if ( 0==strcmp(argmt,"9") ) choice.argmt.rm = ANIMES;
         if (choice.argmt.rm==ALL) usage(prgrm);
         break;
      case EXPECT_HOST_NAME :
          strcpy(choice.argmt.host,argmt); break;
      case EXPECT_LOGIN :
          strcpy(choice.argmt.login,argmt);break;
      default : usage(prgrm);
   }
   *expct=EXPECT_FLAG;
   LOG_OUT();
}


/************************************************************************/
/* void check_consistency (char *prgrm,expect_type expct)               */
/* vrifie la cohrence smantique des arguments et des options         */
/* En cas d'erreur, affichage du message d'erreur, et fin du prgrm      */
/************************************************************************/

void check_consistency (char *prgrm,expect_type expct)
{
   if ( expct&EXPECT_HOST_NAME || expct&EXPECT_LOGIN ) usage(prgrm);
   if ( choice.request&LIST && choice.request>=HOST ) usage(prgrm);
   if ( choice.request&HOST && choice.request!=HOST &&
        choice.request!=(HOST|LONG_FORMAT) && choice.request!=(HOST|SILENT) )
      usage(prgrm);
   if ( choice.request&LOGIN && choice.request!=LOGIN &&
        choice.request!=(LOGIN|LONG_FORMAT) ) usage(prgrm);
   if ( choice.request&DRAW_ROOM && choice.request!=DRAW_ROOM) usage(prgrm);
   
   return;
}



/************************************************************************/
/* choice_type analyze_arguments (int argc,char *argv[])                */
/* analyse les arguments, leur cohrence, et positionne les variables   */
/* de choix                                                             */
/************************************************************************/

void analyze_arguments (int argc,char *argv[])
{
   /* prochain argument attendu : expect */
   expect_type expect = EXPECT_FLAG | EXPECT_ROOM_NBR;
   int i;

   LOG_IN();

   choice.request = LIST;
   choice.argmt.rm = ALL;

   for ( i=1 ; i<argc ; i++ )
      if ( argv[i][0] == '-' ) /* option */
      {
         if ( (expect&EXPECT_FLAG) && (strlen(argv[i])>1) )
            get_option(argv[0],argv[i],&expect);
         else usage(argv[0]);
      }
      else /* argument */
         get_argmt(argv[0],argv[i],&expect);
   check_consistency(argv[0],expect);
   LOG_OUT();
   return;
}


/************************************************************************/
/* void display_main_consoles (info_t *info, char *user, char *mach)    */
/* affiche les principales consoles ("console", "locked", "remote",     */
/* et "remote-locked"), sinon : affiche au moins un terminal.           */
/************************************************************************/

void display_main_consoles (info_t *info, char *user, char *mach)
{
   info_t *current;
   int alrdy=0;

   LOG_IN();

   current = info;
   while (current != NULL)
   {
      if (!strcmp(user,current->user_logged))
         if ( !strcmp(current->console,"console") ||
              !strcmp(current->console,"remote") ||
              !strcmp(current->console,"locked") ||
              !strcmp(current->console,"remote-locked") )
         {
            alrdy = 1;
            printf("   on %s (%s) from %s [%s]\n",mach,
                   current->console,current->host_name,current->log_time);
         }
      current = current->next;
   }

   current = info;
   while ( ( !alrdy ) && ( current != NULL ) )
   {
      if (!strcmp(user,current->user_logged))
      {
         printf("   on %s (%s) from %s [%s]\n",mach,
                current->console,current->host_name,current->log_time);
         alrdy = 1;
      }
      current = current->next;
   }
   LOG_OUT();
   return;
}


/************************************************************************/
/* void display_all_consoles (info_t *info, char *user, char *mach)     */
/* affiche toutes consoles du user pass en parametre 1, et  partir    */
/* du pointeur de liste chainee passe en paramtre 2                   */
/************************************************************************/

void display_all_consoles (info_t *info, char *user, char *mach)
{
   info_t *current;

   LOG_IN();
   current = info;
   while (current != NULL)
   {
      if (!strcmp(user,current->user_logged))
         printf("   on %s (%s) from %s [%s]\n",mach,
                current->console,current->host_name,current->log_time);
      current = current->next;
   }
   LOG_OUT();
   return;
}


/************************************************************************/
/* void host_mode (void)                                                */
/* lance la requete de 'sall' sur une machine particulire              */
/************************************************************************/

void host_mode ()
{
   already_type *stck;
   info_t *info;

   LOG_IN();

   hosts = (interest_t *)allocate_mem(sizeof(interest_t));
   hosts->next = NULL;
   hosts->info = NULL;
   hosts->pthrid = hosts->dsp_thrd = main_pthread_id;
   determine_host(&(hosts->name), choice.argmt.host);

   query();
   printf("\033[4;1m%s :\033[0m\n",hosts->name);
   wait_for_pthrds();

   info = hosts->info;
   if (info != NULL) /* il y des gens loggus sur cette machine */
   {
      init_stack(&stck);

      while ( info != NULL )
      {
         int already;

         already = in_stack(info->user_logged,stck);

         if (!already) /* ce user n'a encore jamais t mentionn */
         {
            add_stack(info->user_logged,&stck);
            if (!strcmp(info->user_logged,"no_reply"))
               printf("\033[4mNo reply from host\033[0m\n");
            else
            {
               printf("\033[1m%s\033[0m (%s) : %s (%s)\n",info->full_name,
                      info->group,info->user_logged,info->nickname);
               if ( !(choice.request&SILENT) )
               {
                  if ( choice.request&LONG_FORMAT )
                     display_all_consoles(info, info->user_logged,
                                          hosts->name);
                  else display_main_consoles(info, info->user_logged,
                                             hosts->name);
               }
            }
         }
         info = info->next;
      }
      empty_stack(&stck);
   }
   else printf("No one logged on ...\n");

   LOG_OUT();
   return;
}


/************************************************************************/
/* void login_mode (void)                                               */
/* recherche un login        qui match 'login_part'                     */
/* la structure already permet de rassembler toutes les machines sur    */
/* lesquelles est loggu 'login_part'                                   */
/************************************************************************/

void login_mode ()
{
   regex_t preg;
   info_t *info;
   interest_t *aux_host;
   already_type *stck;
 
   LOG_IN();
   get_index_machines(ALL_AND_ALL);
   if (regcomp(&preg,choice.argmt.login,REG_EXTENDED|REG_ICASE|REG_NOSUB))
   {
      fprintf(stderr,"\aBad RegExp : %s\n",choice.argmt.login);
      set_them_free();
      exit(7);
   }
   init_stack(&stck);

   wait_for_pthrds();

   aux_host = hosts;
   while ( aux_host != NULL )
   {
      info = aux_host->info;
      while ( info != NULL ) /* il y a q1 de loggu sur cette machine */
      {
         if ( 0 == regexec(&preg,info->user_logged,0,NULL,0) )
         {
            int already;

            already = in_stack(info->user_logged,stck);

            if (!already)
            {
               interest_t *aux_aux_host;

               add_stack(info->user_logged,&stck);
               if (strcmp(info->user_logged,"no_reply"))
               {
                  printf("\033[1m%s\033[0m (%s) : %s (%s)\n",info->full_name,
                         info->group,info->user_logged,info->nickname);
                  if ( choice.request&LONG_FORMAT )
                     display_all_consoles(info, info->user_logged,
                                          aux_host->name);
                  else display_main_consoles(info, info->user_logged,
                                             aux_host->name);
                  aux_aux_host = aux_host->next;
                  while ( aux_aux_host != NULL )
                  {
                     if ( choice.request&LONG_FORMAT )
                        display_all_consoles(aux_aux_host->info,
                           info->user_logged, aux_aux_host->name);
                     else display_main_consoles(aux_aux_host->info,
                             info->user_logged, aux_aux_host->name);
                     aux_aux_host = aux_aux_host->next;
                  }
               }
            }
         } /* il y avait q1 de loggu sur cette machine */
         info = info->next;
      }
      aux_host = aux_host->next;
   }
   empty_stack(&stck);
   regfree(&preg);
   LOG_OUT();
}


/************************************************************************/
/* void get_index_plan (int *first,int *last)                           */
/* cherche la premire et la dernire ligne du plan de la salle rm      */
/* dans le tableau de chaines de caractres 'plan'                      */
/************************************************************************/

void get_index_plan (int *first,int *last)
{
   int r = -1, l = -1;
   LOG_IN();

   while ( (int)r <= (int)(choice.argmt.rm) )
   {
      l++;
      if ( !strcmp(plan[l],"") )
      {
         r++;
         if ( r == choice.argmt.rm ) *first = l+1;
         if ( r == choice.argmt.rm+1 ) *last = l-1;
      }
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void write_nick_grp(info_t *info)                                    */
/* affiche le login ou nickname et le groupe de la personne ventuelle- */
/* ment loggue physiquement sur la machine mach, formattage suivant    */
/* FIELD_SIZE                                                           */
/************************************************************************/

void write_nick_grp(info_t *info)
{
   info_t *aux_info,*console_info;
   char format[MAX_SIZE+1];

   LOG_IN();
   snprintf(format,MAX_SIZE,"%%%d.%ds",FIELD_SIZE,FIELD_SIZE);

   aux_info = info;
   console_info = NULL;
   while ( aux_info != NULL )
   {
      if ( !strcmp(aux_info->console,"console") ||
           !strcmp(aux_info->console,"locked") )
      {
         console_info=aux_info;
         break;
      }
      aux_info = aux_info->next;
   }
   if ( console_info == NULL ) /* personne n'est loggu physiquement */
      printf(format,"");
   else /* q1 est loggu physiquement */
   {
      char s[MAX_SIZE+1];

      snprintf(s,MAX_SIZE,format,console_info->group);
      memcpy(s,console_info->nickname,
             min(FIELD_SIZE-1-strlen(console_info->group),
                 strlen(console_info->nickname)));
      printf(format,s);
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void write_real_name(info_t *info)                                   */
/* affiche le full name de la personne loggue physiquement sur la      */
/* machine numro 'mach'; formattage sur FIELD_SIZE caracteres          */
/************************************************************************/

void write_real_name(info_t *info)
{
   info_t *aux_info,*console_info;
   char format[MAX_SIZE+1];

   LOG_IN();
   snprintf(format,MAX_SIZE,"%%-%d.%ds",FIELD_SIZE,FIELD_SIZE);
   aux_info = info;
   console_info = NULL;
   while ( aux_info != NULL )
   {
      if ( !strcmp(aux_info->console,"console") ||
           !strcmp(aux_info->console,"locked") )
      {
         console_info=aux_info;
         break;
      }
      aux_info = aux_info->next;
   }
   if ( console_info == NULL ) /* personne n'est loggu physiquement */
      printf(format,"");
   else /* q1 est loggu physiquement */
   {
      if ( strcmp(console_info->console,"console") )
         snprintf(format,MAX_SIZE,"[%%-%d.%ds]",FIELD_SIZE-2,FIELD_SIZE-2);
      printf(format,console_info->full_name);
         
   }

   LOG_OUT();
   return;
}


/************************************************************************/
/* void write_host_name(char *name)                                     */
/* affiche le nom de la machine numro 'mach'; centre sur FIELD_SIZE   */
/* caractres                                                           */
/************************************************************************/

void write_host_name(char *name)
{
   char format[MAX_SIZE+1];
   int a,b,c;

   LOG_IN();
   a = strlen(name);
   b = (FIELD_SIZE-a)/2;
   c = FIELD_SIZE-a-b;

   snprintf(format,MAX_SIZE,"%%%d.s%%s%%%d.s",b,c);
   printf(format,"",name,"");

   LOG_OUT();
   return;
}


/************************************************************************/
/* void display_room (void)                                             */
/* affiche les occupants de la salle choice.argmt.rm                    */
/************************************************************************/

void display_room ()
{
   int lgn,col,begin_plan,end_plan;
   interest_t *real_name, *login_group, *macchina;

   LOG_IN();

   get_index_machines(choice.argmt.rm);
   macchina = login_group = real_name = hosts;
   get_index_plan(&begin_plan,&end_plan);

   wait_for_pthrds();

   /* affichage du plan de la salle */
   for (lgn=begin_plan ; lgn<=end_plan ; lgn ++)
   {
      int l;
      l = strlen(plan[lgn]);
      for (col=0 ; col<l ; col++)
      {
         char c;

         c = plan[lgn][col];
         switch (c)
         {
            case '@' : /* afficher le login/nickname et le groupe */
               write_nick_grp(login_group->info);
               login_group = login_group->next;
               break;
            case '^' : /* afficher le real_name */
               write_real_name(real_name->info);
               real_name = real_name->next;
               break;
            case '#' :
               write_host_name(macchina->name);
               macchina = macchina->next;
               break;
            default : printf("%c",c);
         }
      } /* dernire colonne de la ligne ==> carriage return */
      printf("\n");
   }
   LOG_OUT();
}


/************************************************************************/
/* void wait_for_display_line (void)                                    */
/* joint les threads 'display_line' pour afficher la liste en tps rel  */
/************************************************************************/

void wait_for_display_line ()
{
   interest_t *aux_host;

   aux_host = hosts;

   while ( aux_host != NULL )
   {
      if ( aux_host->dsp_thrd != main_pthread_id )
      {
         pthread_join(aux_host->dsp_thrd,NULL);
         aux_host->dsp_thrd = main_pthread_id;
      }
      aux_host = aux_host->next;
   }

   return;
}


/************************************************************************/
/* void display_list (void)                                             */
/* affiche les loggus sous la forme d'une liste (smaphore pour ne     */
/* pas que deux threads essaient d'afficher en meme temps !)            */
/************************************************************************/

void display_list ()
{
   LOG_IN();

   if ( choice.request & ALL_MACHINES )
      get_index_machines(ALL_AND_ALL);
   else
      get_index_machines(ALL);

   wait_for_display_line();

   LOG_OUT();
   return;
}


/************************************************************************/
/* main prgrm                                                           */
/************************************************************************/

int main (int argc,char *argv[])
{
   struct rlimit rlp;

   LOG_IN();

   main_pthread_id = pthread_self();

   /* augmenter le nbr maxi de file descriptors pour que les threads */
   /* puissent ouvrir des sockets comme des brutes */
   getrlimit(RLIMIT_NOFILE,&rlp);
   rlp.rlim_cur = rlp.rlim_max;
   setrlimit(RLIMIT_NOFILE,&rlp);

   analyze_arguments(argc,argv);

   launch_timer();

   switch (choice.request)
   {
      case HOST : case HOST|LONG_FORMAT : case HOST|SILENT :
         host_mode(); break;
      case LOGIN : case LOGIN|LONG_FORMAT :
         login_mode(); break;
      case DRAW_ROOM :
         display_room(); break;
      default :
            display_list();
   }
   set_them_free();
   LOG_OUT();
   return(0);
}

