#!/bin/bash
set -e

###########################################################
### Configuration                                       ###
implementation=direct
ifupdown_class=

############################################################
### Documentation                                        ###
#
# Usage: $(basename "$0") CMD...
# Allowed commands:
#   br-get IFACE:
#      print the name of the OVS brigde of IFACE
#   br-add-if BRIDGE IFACE:
#      add the IFACE interface to the BRIDGE OVS bridge
#   br-del-if BRIDGE IFACE:
#      remove the IFACE interface from the BRIDGE OVS bridge
#
# 'direct' implementation:
# br-get:
#   /usr/bin/ovs-vsctl iface-to-br IFACE
# br-add-if:
#   /usr/bin/ovs-vsctl add-port BRIDGE IFACE
# br-del-if:
#   /usr/bin/ovs-vsctl del-port BRIDGE IFACE
#
# 'ifupdown' implementation (allows one to specify tags to OVS ports)
# br-get:
#   look for the ovs_bridge property with ifquery
# br-add-if:
#   ifup IFACE
# br-del-if:
#   ifdown IFACE
# All ifupdown commands can be restricted to a CLASS (--allow=CLASS)
# if the ifupdown_class is set in the config file
############################################################
### Implementation                                       ###

error() {
   echo 1>&2 "Error: $1"
   sed -e '/^### Documentation *###$/,/^###*$/s/^#\( \|$\)//p;d' "$0" 1>&2
   echo 1>&2

   exit 1
}

case "$1" in
br-get)
   if [ $# != 2 ]; then
      error "wrong number of arguments"
   fi ;;
br-add-if|br-del-if)
   if [ $# != 3 ]; then
      error "wrong number of arguments"
   fi ;;
*) error "bad invocation" ;;
esac

implementation=direct
ifupdown_class=
if [ -r /etc/hostapd/ovs.conf ]; then
   source /etc/hostapd/ovs.conf
fi

case "$implementation" in
direct) ;;
ifupdown) ;;
*) error "Bad implementation ('$implementation' is unknown)" ;;
esac

direct-br-get() {
   exec /usr/bin/ovs-vsctl iface-to-br "$@"
}

direct-br-add-if() {
   exec /usr/bin/ovs-vsctl add-port "$@"
}

direct-br-del-if() {
   exec /usr/bin/ovs-vsctl del-port "$@"
}

ifupdown-br-get() {
   local IFACE="$1"
   local BRIDGE
   local ALLOW=
   if [ -n "$ifupdown_class" ]; then
      ALLOW="--allow=$ifupdown_class"
   fi
   BRIDGE="$(usr/sbin/ifquery "$ALLOW" "$IFACE" \
      | sed -e 's/^ovs_bridge: //p;d')"
   if [ -z "$BRIDGE" ]; then
      return 1
   fi
   echo "$BRIDGE"
}

ifupdown-br-add-if() {
   local BRIDGE="$1"
   local IFACE="$2"
   local ALLOW=
   if [ -n "$ifupdown_class" ]; then
      ALLOW="--allow=$ifupdown_class"
   fi
   if [ -z "$(/usr/sbin/ifquery -l "$ALLOW" "$IFACE")" ]; then
      return 1
   fi
   exec ifup "$ALLOW" "$IFACE"
}

ifupdown-br-del-if() {
   local BRIDGE="$1"
   local IFACE="$2"
   local ALLOW=
   if [ -n "$ifupdown_class" ]; then
      ALLOW="--allow=$ifupdown_class"
   fi
   if [ -z "$(/usr/sbin/ifquery -l "$ALLOW" "$IFACE")" ]; then
      return 1
   fi
   exec ifdown "$ALLOW" "$IFACE"
}

eval "$implementation"-'"$@"'
